/**
 * Функция для указания типа payload в `getActionCreator`.
 * Результат функции должен использоваться только
 * в качестве второго аргумента в `getActionCreator`
 *
 * @example
 * // payload в экшен креатора типа string
 * getActionCreator('MY_TYPE', payload<string>())
 */
export function payload<P>() {
    return null as any as P;
}

/**
 * Функция используется для сохранения типа строкового значения
 * TypeScript всегда расширяет тип переменной, например
 * в `const MY_ACTION = '@MY_ACTION'`, MY_ACTION имеет тип string
 * Функция action позволит сохранить тип '@MY_ACTION'
 * `const MY_ACTION = action('@MY_ACTION')`, MY_ACTION имеет тип '@MY_ACTION'
 *
 * Используется при вынесении типа экшена в переменную.
 *
 * @param {string} actionType
 */
export function action<T extends string>(actionType: T): T {
    return actionType;
}

/**
 * Функция возвращает action-creator с указаным type,
 * и строгим типом для payload. Используется для строгой типизации.
 * Для получения типа action используется `ReturnType`
 *
 * @example
 * const myTypeActionCreator = getActionCreator('MyType')
 * // type ActionType = ReturnType<typeof myTypeActionCreator>
 * // ActionType имеет тип { type: 'MyType', payload: undefined }
 *
 * myTypeActionCreator() // { type: 'MyType' }
 *
 * @example
 * // Можно передать тип для payload
 * const myTypeActionCreator = getActionCreator('MyType', payload<string>())
 * // type ActionType = ReturnType<typeof myTypeActionCreator>
 * // ActionType имеет тип { type: 'MyType', payload: string }
 *
 * myTypeActionCreator(123)
 * // -----------------^^^
 * // // error: [ts] Argument of type '123' is not assignable to parameter of type 'string'.
 *
 * myTypeActionCreator('123') // { type: 'MyType', payload: '123' }
 *
 * @param {string} type
 * @param {P} [payload] - аргумент для наследования типа payload
 * @template P
 */
export function getActionCreator<T extends string>(
    type: T,
): () => {type: T; payload: undefined};
export function getActionCreator<T extends string, P>(
    type: T,
    payloadType?: P,
): (payload: P) => {type: T; payload: P};
export function getActionCreator<T extends string, P>(
    type: T,
    _payloadType?: P,
) {
    return function (payload?: P) {
        return {type, payload};
    };
}
