import {parse as queryStringParse} from 'query-string';

import {TQueryType, TQueryTypeWithArray} from 'types/common/IQueryParams';

/**
 * Возвращает распаршенный query из location.search.
 *
 * @param location
 * @param [allowArrays] - параметр, который настраивает, может ли значения ключа быть массивом.
 *                        Если выставлено значение false, то для массивов вернется первый элемент.
 *                        Основная необходимость в том, что в подавляющем числе случаев нам не нужны значения массивы,
 *                        поэтому не хочется везде делать проверку на типы, что значение не массив.
 * @example
 * getQueryByLocation({ search: 'a=foo&b=bar' }) === { a: 'foo', b: 'bar' }
 * getQueryByLocation({ search: 'a=foo&b=bar&b=baz' }) === { a: 'foo', b: ['bar', 'baz'] }
 * getQueryByLocation({ search: 'a=foo&b=bar&b=baz' }, true) === { a: 'foo', b: ['bar', 'baz'] }
 * getQueryByLocation({ search: 'a=foo&b=bar&b=baz' }, false) === { a: 'foo', b: 'bar' }
 */
function getQueryByLocation(
    location: {search: string} | undefined,
    allowArrays?: true,
): Record<string, TQueryTypeWithArray>;
function getQueryByLocation(
    location: {search: string} | undefined,
    allowArrays: false,
): Record<string, TQueryType>;
function getQueryByLocation(
    location: {search: string} | undefined,
    allowArrays: boolean,
): Record<string, TQueryType | TQueryTypeWithArray>;
function getQueryByLocation(
    location: {search: string} | undefined,
    allowArrays = true,
): Record<string, TQueryType> | Record<string, TQueryTypeWithArray> {
    if (!location) {
        return {};
    }

    const {search = ''} = location;

    const parsedQuery = queryStringParse(search);

    if (allowArrays) {
        return parsedQuery;
    }

    return Object.entries(parsedQuery).reduce<{
        [key: string]: TQueryType;
    }>(
        (parsedQueryWithoutArrays, [queryKey, queryValue]) => ({
            ...parsedQueryWithoutArrays,
            [queryKey]: Array.isArray(queryValue) ? queryValue[0] : queryValue,
        }),
        {},
    );
}

export default getQueryByLocation;
