import {useCallback, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {SUBSCRIPTION_CONFIG_LIFETIME} from 'constants/subscription';

import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {IPromoCodeForSubscription} from 'server/api/NotifierApi/types';

import {setSubscriptionConfig} from 'reducers/common/subscription/promoCodeInfo/actions';

import {subscriptionConfigSelector} from 'selectors/common/subscription';

import {getNow} from 'utilities/dateUtils';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useAsync, TUseAsyncState} from 'utilities/hooks/useAsync';

import {notifierBrowserProvider} from 'serviceProvider/notifier/notifierBrowserProvider';

export function useAsyncSubscriptionConfig(
    vertical: ESubscriptionVerticalName,
): TUseAsyncState<Nullable<IPromoCodeForSubscription>> {
    const dispatch = useDispatch();
    const userInfo = useUserInfo();
    const commonConfig = useSelector(subscriptionConfigSelector);
    const config = commonConfig[vertical];
    const email = isAuthUser(userInfo) ? userInfo.contacts.email : undefined;

    const fetcher = useCallback(async () => {
        // Если данные были получены ранее и не протухли - возвращаем их
        if (
            config &&
            config.timestamp + SUBSCRIPTION_CONFIG_LIFETIME > getNow()
        ) {
            return config?.data;
        }

        // Если данных нет или они протухли - идём в ручку и сохраняем ответ
        // в async state как хранилище данных для интерфейса и в store как в кеш
        const updatedConfig = await notifierBrowserProvider.getPromoConfig(
            vertical,
            email,
        );

        dispatch(setSubscriptionConfig([vertical, updatedConfig]));

        return updatedConfig?.promoCode || null;
    }, [dispatch, config, vertical, email]);
    const [state, asyncFetcher] = useAsync(fetcher);

    useEffect(() => {
        asyncFetcher();
    }, [asyncFetcher]);

    return state;
}
