import {useState, useRef, useCallback} from 'react';

import {tupple} from 'utilities/tupple';
import {unknownToError} from 'utilities/error';

export type TUseAsyncState<Data> =
    | {loading: true; data: null; error: null}
    | {loading: false; data: Data; error: null}
    | {loading: false; data: null; error: Error};

export function useAsync<Data, Arguments extends any[]>(
    fn: (...args: Arguments) => Promise<Data>,
) {
    const idRef = useRef(0);
    const [state, setState] = useState<TUseAsyncState<Data>>({
        loading: true,
        data: null,
        error: null,
    });

    const runAsync = useCallback(
        async function runAsync(...args: Arguments) {
            const expectingId = ++idRef.current;

            setState({loading: true, data: null, error: null});

            try {
                const data = await fn(...args);

                if (idRef.current === expectingId) {
                    setState({loading: false, data, error: null});
                }
            } catch (error) {
                if (idRef.current === expectingId) {
                    setState({
                        loading: false,
                        data: null,
                        error: unknownToError(error),
                    });
                }
            }
        },
        [fn],
    );

    return tupple(state, runAsync);
}
