import {useRef, useState, useLayoutEffect, MutableRefObject} from 'react';

export interface ISize {
    width: number;
    height: number;
}

export const DEFAULT_ELEMENT_SIZE: ISize = {
    width: 0,
    height: 0,
};

/* Calculate element size (only mount) */
// Важно: ref.current не должен ссылаться на разные ноды в течение жизни хука
// Если нужно поддержать такой функционал, надо переделать, чтобы пропс принимал сам нод, а не реф
export const useElementSize = <Id, Element extends HTMLElement>({
    id,
    watchForChanges,
}: {
    id?: Id;
    watchForChanges?: boolean;
}): {
    ref: MutableRefObject<Element | null>;
    size: ISize;
    isMeasured: boolean;
} => {
    const ref = useRef<Element | null>(null);
    const [size, setElementSize] = useState<ISize>(DEFAULT_ELEMENT_SIZE);
    const [isMeasured, setIsMeasured] = useState(false);

    useLayoutEffect(() => {
        const {current: currentNode} = ref;

        if (!currentNode) {
            return;
        }

        const setSize = (): void => {
            const {width, height} = currentNode.getBoundingClientRect();

            setElementSize({width, height});

            setIsMeasured(true);
        };

        setSize();

        if (watchForChanges) {
            const observer = new ResizeObserver(setSize);

            observer.observe(currentNode);

            return (): void => observer.disconnect();
        }
    }, [id, watchForChanges]);

    return {
        ref,
        size,
        isMeasured,
    };
};
