import {useEffect} from 'react';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {usePageLoadInfo} from 'utilities/hooks/usePageLoadInfo';
import {useWasNavigation} from 'utilities/hooks/useWasNavigation';
import {timeout} from 'utilities/timeout';

const DEFAULT_TIMEOUT_PROPORTION = 1.5;
const MAX_WAIT_AFTER_LOAD_MS = 5 * 1000;

export interface ILoadDynamicResourceOptions {
    timeoutProportion?: number;
    maxProportionTimeout?: number;
    timeoutAfterNavigation?: number;
}

/**
 * Хук, определяющий когда необходимо загрузить динамический ресурс
 * Необходим для того, чтобы не загружать лишние ресурсы во время загрузки
 * страницы и какое-то время после, что в свою очередь улучшает UX и оценки
 * в Lighthouse
 *
 * Если произошла навигация во время загрузки страницы или во время
 * таймаута, то ресурс загрузится сразу же, так как интерактивность
 * уже наступила
 *
 * @param options.timeoutProportion - Пропорция по отношению к длительности загрузки
 * страницы через какое время после загрузки страницы загрузить ресурс
 *
 * Пример: пропорция - 2, загрузка страницы длилась 3 секунды, значит
 * загрузка ресурса должна произойти через 6 секунд после загрузки страницы
 *
 * Должно быть константным, default value: 1.5
 *
 * @param options.maxProportionTimeout - максимальный таймаут,
 * через который нужно загрузить ресурс
 *
 * Пример: таймаут - 3000, загрузка страницы длилась 7 секунды, пропорция - 2, значит
 * загрузка ресурса должна произойти через 3, а не 14 секунд после загрузки страницы
 *
 * Должно быть константным, default value: 5000
 *
 * @param options.timeoutAfterNavigation - Таймаут после которого загрузится ресурс
 * по прошествии первой навигации. Если ресурс не является необходимым для рендера,
 * то следует сюда передать ненулевое значение, чтобы отложить загрузку после рендера
 *
 * Смысл параметра: если ресурс пытаются загрузить при первой загрузке, этот параметр
 * ни на что не влияет, но если ресурс загружается на странице, на которую снавигировались
 * с другой  в пределах одного визита, то все основные ресурсы уже давно загружены и можно
 * быстрее загрузить этот ресурс
 *
 * Должно быть константным, default value: 0
 */
export const useNeedToLoadDynamicResource = (
    options: ILoadDynamicResourceOptions = {},
): boolean => {
    const {
        timeoutProportion = DEFAULT_TIMEOUT_PROPORTION,
        timeoutAfterNavigation = 0,
        maxProportionTimeout = MAX_WAIT_AFTER_LOAD_MS,
    } = options;
    const {value: needToLoadDynamicResource, setTrue: loadDynamicResource} =
        useBoolean(false);
    const wasNavigation = useWasNavigation();
    const pageLoadInfo = usePageLoadInfo();

    /**
     * По коду loadDynamicResource может вызваться несколько раз, но
     * это спровоцирует реальные изменения только один раз
     */
    useEffect(() => {
        if (wasNavigation) {
            return timeout(loadDynamicResource, timeoutAfterNavigation);
        }
    }, [loadDynamicResource, timeoutAfterNavigation, wasNavigation]);

    useEffect(() => {
        if (wasNavigation || !pageLoadInfo.loaded) {
            return;
        }

        const timeSinceLoad = Date.now() - pageLoadInfo.loadedAt;
        const waitTimeAfterLoad = Math.min(
            maxProportionTimeout,
            pageLoadInfo.loadDuration * timeoutProportion,
        );

        return timeout(
            loadDynamicResource,
            Math.max(0, waitTimeAfterLoad - timeSinceLoad),
        );
    }, [
        loadDynamicResource,
        maxProportionTimeout,
        pageLoadInfo,
        timeoutAfterNavigation,
        timeoutProportion,
        wasNavigation,
    ]);

    return needToLoadDynamicResource;
};
