import {useCallback, useEffect, useRef} from 'react';
import {createPath, LocationListener} from 'history';
import uniqueId from 'lodash/uniqueId';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import browserHistory, {
    IBrowserHistoryState,
} from 'utilities/browserHistory/browserHistory';

enum EModalCloseReason {
    UI,
    BROWSER_BACK,
}

export interface ISupportHistoryBackProps {
    isVisible: boolean;
    close?: Function;
    onHistoryBack?: Function;
    popHistoryOnUnmount?: boolean;
}

/**
 * Хук, для закрытия модалов и попапов на браузерный back.
 *
 * При открытии каждого модала делает push в историю браузера (history), чтобы при браузерном "Назад" закрыть модал и вернуться в исходное состояние.
 * При закрытии модала в UI вызывается программный `history.goBack`, чтобы убрать добавленный для модала state из history и вернуться в состояние до открытия модала.
 *
 * Если при открытом модале на странице происходил push в history, не связанный с модалом, то вернуться по history в начальное состояние нельзя.
 * Тогда при закрытии модала произойдет обычный replace.
 */
export default function useSupportHistoryBack({
    isVisible,
    close,
    onHistoryBack,
    popHistoryOnUnmount = true,
}: ISupportHistoryBackProps): void {
    const {isMobile} = useDeviceType();

    const unsubscribeHistoryListenRef = useRef<null | (() => void)>(null);

    const modalIdRef = useRef(uniqueId('layerId_'));
    const lastPathRef = useRef<string>();
    const lastModalIdRef = useRef<string>();
    const modalCloseReasonRef = useRef<EModalCloseReason | null>(null);
    const wasChangedNotModalStateRef = useRef(false);

    const unsubscribe = useCallback(() => {
        if (unsubscribeHistoryListenRef.current) {
            unsubscribeHistoryListenRef.current();
            unsubscribeHistoryListenRef.current = null;
        }
    }, []);

    const handleHistoryChange: LocationListener<IBrowserHistoryState> =
        useCallback(
            (location, action) => {
                if (
                    action === 'POP' &&
                    lastModalIdRef.current === modalIdRef.current
                ) {
                    unsubscribe();

                    /*
                     * Вызываем close только для браузерного назад. Если "назад" произошел из-за вызова goBack
                     * при закрытии модала в UI, вызывать close уже не нужно
                     */
                    if (modalCloseReasonRef.current !== EModalCloseReason.UI) {
                        modalCloseReasonRef.current =
                            EModalCloseReason.BROWSER_BACK;
                        close?.();
                        onHistoryBack?.();
                    }

                    if (
                        lastPathRef.current &&
                        lastPathRef.current !== createPath(location)
                    ) {
                        browserHistory?.replace(lastPathRef.current);
                    }
                } else if (action === 'PUSH' || action === 'REPLACE') {
                    lastPathRef.current = createPath(location);

                    /* Отслеживаем и фиксируем события push/replace в history не связанные с модалом */
                    if (!location.state?.modalId) {
                        wasChangedNotModalStateRef.current = true;
                    }
                }

                /* Отслеживаем и фиксируем событие push в history не связанное с модалом */
                if (action === 'PUSH' && !location.state?.modalId) {
                    wasChangedNotModalStateRef.current = true;
                }

                if (location.state?.modalId) {
                    lastModalIdRef.current = location.state.modalId;
                }
            },
            [close, onHistoryBack, unsubscribe],
        );

    const subscribe = useCallback(() => {
        if (browserHistory && !unsubscribeHistoryListenRef.current) {
            unsubscribeHistoryListenRef.current =
                browserHistory.listen(handleHistoryChange);
        }
    }, [handleHistoryChange]);

    const handleShowModal = useCallback(() => {
        if (!browserHistory) {
            return;
        }

        const path = createPath(browserHistory.location);
        const newHistoryState: IBrowserHistoryState = {
            ...browserHistory.location.state,
            modalId: modalIdRef.current,
            scrollTop: null,
        };

        modalCloseReasonRef.current = null;
        wasChangedNotModalStateRef.current = false;

        subscribe();
        browserHistory.push(path, newHistoryState);
    }, [subscribe]);

    const handleHideModal = useCallback(() => {
        if (
            !browserHistory ||
            modalCloseReasonRef.current === EModalCloseReason.BROWSER_BACK
        ) {
            return;
        }

        modalCloseReasonRef.current = EModalCloseReason.UI;

        if (wasChangedNotModalStateRef.current) {
            unsubscribe();
        } else {
            browserHistory.goBack();
        }
    }, [unsubscribe]);

    /* Effects */

    useEffect(() => {
        if (!isMobile) {
            return;
        }

        if (isVisible) {
            handleShowModal();
        }
    }, [handleShowModal, isMobile, isVisible]);

    useEffect(() => {
        if (!isMobile) {
            return;
        }

        return (): void => {
            /* Если isVisible сменяется с true на false и закрытие модала вызовано не браузерным back */
            if (popHistoryOnUnmount && isVisible) {
                handleHideModal();
            }
        };
    }, [handleHideModal, isMobile, isVisible, popHistoryOnUnmount]);

    /* Отписываемся если модал анмаунтится без закрытия */
    useEffect(
        () => (): void => {
            unsubscribe();
        },
        [unsubscribe],
    );
}
