import {useCallback, useEffect, useState} from 'react';
import {useLocation} from 'react-router-dom';

interface IUseTimeOnTabParams {
    timeout: number; // Количество миллисекунд, после которых нужно вызвать callback
    callback: () => void;
}

/** Вызывет callback, после нахождения на странице timeout миллисеунд.
 * Учитываеет переключение на другие вкладки и не считает время, проведенное на них.
 * Пример кейса, когда может быть полезно: отправить событие в метрику,
 *    если пользователь находится на странице дольше 2-х минут. */
export function useTimeOnTab(params: IUseTimeOnTabParams): void {
    const {timeout, callback} = params;

    const [switchBackTimestamp, setSwitchBackTimestamp] = useState<number>(0);
    const [timeSpentOnTab, setTimeSpentOnTab] = useState(0);
    const [eventWasSent, setEventWasSent] = useState(false);
    const [timeOutId, setTimeOutId] = useState<ReturnType<
        typeof setTimeout
    > | null>(null);
    const {pathname} = useLocation();

    const setTimeoutOnTime = useCallback(
        (time: number) => {
            const newTimeOutId = setTimeout(() => {
                if (document.location.pathname === pathname) {
                    callback();
                }
                setEventWasSent(true);
            }, time);

            setTimeOutId(newTimeOutId);

            const now = Date.now();

            setSwitchBackTimestamp(now);
        },
        [
            setEventWasSent,
            setTimeOutId,
            setSwitchBackTimestamp,
            callback,
            pathname,
        ],
    );

    const onVisibilityChange = useCallback(() => {
        if (document.hidden) {
            const now = Date.now();
            const newTimeSpentOnTab =
                timeSpentOnTab + (now - switchBackTimestamp);

            setTimeSpentOnTab(newTimeSpentOnTab);

            if (timeOutId) {
                clearTimeout(timeOutId);
            }
        } else {
            if (!eventWasSent) {
                setTimeoutOnTime(timeout - timeSpentOnTab);
            }
        }
    }, [
        switchBackTimestamp,
        timeSpentOnTab,
        setTimeSpentOnTab,
        eventWasSent,
        timeOutId,
        timeout,
        setTimeoutOnTime,
    ]);

    useEffect(() => {
        document.addEventListener('visibilitychange', onVisibilityChange);

        if (!timeOutId && !eventWasSent) {
            setTimeoutOnTime(timeout);
        }

        return (): void => {
            document.removeEventListener(
                'visibilitychange',
                onVisibilityChange,
            );
        };
    }, [
        onVisibilityChange,
        eventWasSent,
        timeOutId,
        setTimeoutOnTime,
        timeout,
    ]);
}
