import {useCallback, useEffect, useRef} from 'react';

import {useBoolean} from 'utilities/hooks/useBoolean';

export default function useTooltipHover(delay = 400): {
    isHovered: boolean;
    handleMouseEnter(): void;
    handleMouseLeave(): void;
} {
    const {
        value: isHovered,
        setTrue: showTooltip,
        setFalse: hideTooltip,
    } = useBoolean(false);

    const showTimerRef = useRef<number>();
    const wasScrollListenerSetRef = useRef(false);

    const clearShowTimeout = useCallback(() => {
        if (showTimerRef.current) {
            window.clearTimeout(showTimerRef.current);
        }
    }, [showTimerRef]);

    const scrollListener = useCallback(() => {
        clearShowTimeout();
        hideTooltip();

        document.removeEventListener('scroll', scrollListener);

        wasScrollListenerSetRef.current = false;
    }, [clearShowTimeout, hideTooltip]);

    const handleMouseLeave = useCallback(() => {
        clearShowTimeout();
        hideTooltip();

        if (wasScrollListenerSetRef.current) {
            document.removeEventListener('scroll', scrollListener);

            wasScrollListenerSetRef.current = false;
        }
    }, [clearShowTimeout, hideTooltip, scrollListener]);

    const handleMouseEnter = useCallback(() => {
        showTimerRef.current = window.setTimeout(showTooltip, delay);

        if (!wasScrollListenerSetRef.current) {
            wasScrollListenerSetRef.current = true;

            document.addEventListener('scroll', scrollListener);
        }
    }, [showTooltip, delay, scrollListener]);

    useEffect(
        () => (): void => {
            clearShowTimeout();
        },
        [clearShowTimeout],
    );

    return {
        isHovered,
        handleMouseEnter,
        handleMouseLeave,
    };
}
