import {useCallback, useEffect, useRef} from 'react';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {getNow} from 'utilities/dateUtils';

const EXIT_PAGE_THRESHOLD = 100;

export function useLeaving(delay: number = 0): boolean {
    const {value, setTrue, setFalse} = useBoolean(false);
    const trackingStart = useRef<number>(getNow());
    const mouseYPosition = useRef<number>(EXIT_PAGE_THRESHOLD);

    const handleMouseMove = useCallback((event: MouseEvent) => {
        mouseYPosition.current = event.clientY;
    }, []);

    const handleDocumentLeave = useCallback(() => {
        const trackingTime = getNow() - trackingStart.current;
        const isAvailableForTracking = trackingTime > delay;

        if (
            isAvailableForTracking &&
            mouseYPosition.current < EXIT_PAGE_THRESHOLD
        ) {
            setTrue();
        }
    }, [setTrue, delay]);

    useEffect(() => {
        document.addEventListener('mouseenter', setFalse);
        document.addEventListener('mouseleave', handleDocumentLeave);
        document.addEventListener('mousemove', handleMouseMove);

        return (): void => {
            document.removeEventListener('mouseenter', setFalse);
            document.removeEventListener('mouseleave', handleDocumentLeave);
            document.removeEventListener('mousemove', handleMouseMove);
        };
    }, [setFalse, handleMouseMove, handleDocumentLeave]);

    return value;
}
