import {
    getStorageItem,
    removeStorageItem,
    saveStorageItem,
} from 'utilities/storageUtils';

interface IDataContainer<T> {
    data: T;
    timestamp: number;
}

/**
 * Сохраняет в localStorage с истечением срока актуальности сохраненных данных
 * При чтении устаревших данных вернет null
 */
export class StorageSaverWithExpiration<T> {
    private readonly expiration: number;

    private readonly storageKey: string;

    private readonly storage: Storage | undefined;

    /**
     * @param storageKey имя ключа в localStorage
     * @param expiration время протухания в миллисекундах
     */
    constructor(storageKey: string, expiration: number) {
        this.storageKey = storageKey;
        this.expiration = expiration;

        this.storage =
            typeof window === 'undefined' ? undefined : window.localStorage;
    }

    save(data: T): void {
        const container: IDataContainer<T> = {
            data: data,
            timestamp: new Date().getTime(),
        };

        saveStorageItem(this.storage, this.storageKey, container);
    }

    read(): T | null {
        const savedDataContainer: IDataContainer<T> | undefined =
            getStorageItem(this.storage, this.storageKey);

        if (!savedDataContainer) {
            return null;
        }

        let data: T | undefined;
        let timestamp: number | undefined;

        try {
            data = savedDataContainer.data;
            timestamp = savedDataContainer.timestamp;
        } catch (e) {
            return null;
        }

        if (data === undefined || !timestamp) {
            return null;
        }

        const currentTime = new Date().getTime();

        const expired = currentTime - timestamp > this.expiration;

        if (expired) {
            removeStorageItem(this.storage, this.storageKey);

            return null;
        }

        return data;
    }
}
