import browserHistory from 'utilities/browserHistory/browserHistory';

export interface IPageNotLoadedInfo {
    loaded: false;
    loadDuration: null;
    loadedAt: null;
}

export interface IPageLoadedInfo {
    loaded: true;
    loadDuration: number;
    loadedAt: number;
}

export type TPageLoadInfo = IPageNotLoadedInfo | IPageLoadedInfo;

export type TOnLoadCallback = (pageLoadInfo: IPageLoadedInfo) => unknown;
export type TOnFirstNavigationCallback = () => unknown;
export type TUnregisterCallback = () => void;

class PageManager {
    pageLoadInfo: TPageLoadInfo = {
        loaded: false,
        loadDuration: null,
        loadedAt: null,
    };
    wasNavigation: boolean = false;

    private initialized: boolean = false;
    private initialPagePathname: string | undefined;
    private loadCallbacks: Set<TOnLoadCallback> = new Set();
    private firstNavigationCallbacks: Set<TOnFirstNavigationCallback> =
        new Set();

    addOnLoadCallback(callback: TOnLoadCallback): TUnregisterCallback {
        this.loadCallbacks.add(callback);

        return (): void => {
            this.loadCallbacks.delete(callback);
        };
    }

    addOnFirstNavigationCallback(
        callback: TOnFirstNavigationCallback,
    ): TUnregisterCallback {
        this.firstNavigationCallbacks.add(callback);

        return (): void => {
            this.firstNavigationCallbacks.delete(callback);
        };
    }

    init(): void {
        if (this.initialized) {
            return;
        }

        this.initialized = true;
        this.initialPagePathname = browserHistory?.location.pathname;

        const removeListener = browserHistory?.listen(location => {
            if (location.pathname === this.initialPagePathname) {
                return;
            }

            removeListener?.();

            this.wasNavigation = true;

            this.firstNavigationCallbacks.forEach(callback => {
                callback();
            });
        });

        if (document.readyState === 'complete') {
            this.onLoad();

            return;
        }

        document.addEventListener('readystatechange', () => {
            if (document.readyState === 'complete') {
                this.onLoad();
            }
        });
    }

    private onLoad(): void {
        this.pageLoadInfo = {
            loaded: true,
            loadDuration: performance.now(),
            loadedAt: Date.now(),
        };

        for (const loadCallback of this.loadCallbacks) {
            loadCallback(this.pageLoadInfo);
        }
    }
}

export default new PageManager();
