export type TStateValue =
    | string
    | number
    | boolean
    | undefined
    | null
    | {[key: string]: TStateValue}
    | {[key: number]: TStateValue};

export type TState = Record<string, TStateValue>;

export const prev = '_prev';
export const current = '_current';

function typeOf(value: unknown): string {
    if (value === null) {
        return 'null';
    }

    if (Array.isArray(value)) {
        return 'array';
    }

    return typeof value;
}

/**
 * Возвращает разницу между двумя объектами.
 * @param a - объект "до"
 * @param b - объект "после"
 * @param maxElements - максимальное количество элементов объекта для сравнения в одном свойстве. Если два объекта
 * будут различаться на более чем maxElements элементов в одном свойстве, то дальнейшее сравнение этого свойства
 * прекратится и будет отображены различия только maxElements.
 */
export default function diffState(
    a: TStateValue,
    b: TStateValue,
    maxElements?: number,
): TState | undefined {
    const result: TState = {};
    const typeA = typeOf(a);
    const typeB = typeOf(b);
    const maxDiff =
        typeof maxElements === 'number' && maxElements > 1 ? maxElements : 0;

    if (typeA !== typeB) {
        return {
            [prev]: a,
            [current]: b,
        };
    }
    // Далее одинаковые типы

    if (a === b) {
        return;
    }
    // Далее разные значения

    if (typeA !== 'array' && typeA !== 'object') {
        return {
            [prev]: a,
            [current]: b,
        };
    }

    // Для ts
    if (
        typeof a !== 'object' ||
        typeof b !== 'object' ||
        a === null ||
        b === null
    ) {
        return;
    }

    // Сравнение объектов одного типа
    let elements = 0;
    const keys = new Set(Object.keys(a).concat(Object.keys(b)));

    for (const key of keys) {
        if (maxDiff && elements > maxDiff) {
            break;
        }

        // @ts-ignore
        // Из-за особенностей js (Object.keys() и для массивов вернет string[]) и потенциально высокой нагрузке
        // на данный код, лучше не писать дополнительный код, чтобы объяснить ts что тут все нормально
        const diff = diffState(a[key], b[key], maxElements);

        if (diff) {
            result[key] = diff;
            elements++;
        }
    }

    return Object.keys(result).length ? result : undefined;
}
