import pickBy from 'lodash/pickBy';
import isEmpty from 'lodash/isEmpty';
import {stringify, parseUrl} from 'query-string';

import {TLD} from 'constants/tld';
import {PRODUCTION_PROJECT_URL} from 'constants/common';

import {IInternalUrlOptions, IUrlQuery} from 'utilities/url/types';

import {getStoredSrcParams} from 'utilities/srcParams/srcParams';

const DEFAULT_INTERNAL_URL_OPTIONS: IInternalUrlOptions = {
    withOrigin: false,
    tld: TLD.RU,
    trailingSlash: true,
    filterNull: false,
};

/**
 * Возвращает origin проекта.
 *
 * @param {TLD} [tld] - TLD для origin. Если не задано, то будет `ru`.
 *
 * @return {string}
 */
export function getProjectUrl(tld: TLD = TLD.RU): string {
    return `https://travel.yandex.${tld}`;
}

export function formatQuery(
    query: IUrlQuery,
    {filterNull = false, encode = true} = {},
): string {
    const filteredQuery = filterNull
        ? // eslint-disable-next-line no-eq-null
          pickBy(query, value => value != null)
        : query;

    return isEmpty(filteredQuery) ? '' : stringify(filteredQuery, {encode});
}

/**
 * Формирует внутреннюю ссылку для проекта.
 * Правильно расставляет слеши в конце ссылки.
 * Правильно формирует query-параметры.
 *
 * @param path - Путь ссылки.
 * @param [query] - Объект query-параметров.
 * @param [options] - Опции формирования URL.
 * @param [fragment] - Объект fragment-параметров (hash).
 */
export function internalUrl(
    path: string,
    query?: IUrlQuery | null,
    options?: IInternalUrlOptions | null,
    fragment?: IUrlQuery,
    hash?: string,
): string {
    const preparedOption = {...DEFAULT_INTERNAL_URL_OPTIONS, ...options};
    const srcParams = getStoredSrcParams();
    const extendedQuery = srcParams ? {...query, srcParams} : query;
    let url = path;

    if (!url.startsWith('/')) {
        url = `/${url}`;
    }

    if (!url.endsWith('/') && preparedOption.trailingSlash) {
        url = `${url}/`;
    }

    if (preparedOption.withOrigin) {
        url = getProjectUrl(preparedOption.tld) + url;
    }

    if (extendedQuery) {
        const queryString = formatQuery(extendedQuery, {
            filterNull: preparedOption.filterNull,
        });

        url = `${url}${queryString.length > 0 ? `?${queryString}` : ''}`;
    }

    if (fragment) {
        const hashString = formatQuery(fragment, {
            filterNull: preparedOption.filterNull,
            encode: false,
        });

        url = `${url}${hashString.length > 0 ? `#${hashString}` : ''}`;
    } else if (hash) {
        url = `${url}#${hash}`;
    }

    return url;
}

/**
 *
 * Формирует внешнюю ссылку с query-параметрами.
 *
 * @param url - URL ссылки.
 * @param query - Объект query-параметров.
 * @param filterNull - Фильтровать null значения в query-параметрах.
 */
export function externalUrl(
    url: string,
    query?: IUrlQuery,
    {filterNull = false} = {},
): string {
    if (query) {
        const queryString = formatQuery(query, {filterNull});

        return `${url}${queryString.length > 0 ? `?${queryString}` : ''}`;
    }

    return url;
}

/**
 * Заменяет или добавляет query-параметры
 * @param url - URL ссылки
 * @param needQueryParams - query-параметры
 */
export function replaceQuery(
    url: string,
    needQueryParams: IUrlQuery,
): IUrlQuery {
    const {query} = parseUrl(url);

    return {
        ...query,
        ...needQueryParams,
    };
}

export function getAbsoluteUrl(path: string, search: string = ''): string {
    return `${PRODUCTION_PROJECT_URL}${path}${search}`;
}
