import _set from 'lodash/set';
import _get from 'lodash/get';
import _has from 'lodash/has';
import {ValidationErrors} from 'final-form';

import {
    IFormValidationInfo,
    IFieldGroup,
    IFormValues,
    TField,
} from 'types/common/validation/form';
import {
    TValidations,
    IValidationOptions,
    IFormValidationData,
} from 'types/common/validation/validation';

import getValidationError from 'utilities/validation/form/getValidationError';
import filterDependentValidations from 'utilities/validation/form/filterDependentValidations';

interface IValidateFieldGroup {
    validationGroup: IFieldGroup;
    formData: IFormValidationData & {
        groupValues: IFormValues;
    };
    errors: ValidationErrors;
    options: IValidationOptions;
}

function reduceValidationErrors(
    value: unknown,
    validations: TValidations[],
    formData: IFormValidationData,
): string | undefined {
    for (let i = 0; i < validations.length; i++) {
        const validation = validations[i];
        const error = getValidationError(value, validation, formData);

        if (error) {
            return error;
        }
    }

    return undefined;
}

function getFieldError(
    value: unknown,
    field: TField,
    formData: IValidateFieldGroup['formData'],
    options: IValidationOptions,
): string | undefined {
    const {groupValues} = formData;

    const fieldRules = options.isSubmit
        ? field.validation?.submit
        : field.validation?.blur;
    const error = getValidationError(value, fieldRules, formData);

    if (!error && field.dependentValidations) {
        const blurDependValidations = filterDependentValidations(
            groupValues,
            field.dependentValidations,
            formData,
            options,
        );

        return reduceValidationErrors(value, blurDependValidations, formData);
    }

    return error;
}

function validateGroupByValues({
    validationGroup,
    formData,
    errors,
    options,
}: IValidateFieldGroup): void {
    const {groupValues, groupPath} = formData;

    validationGroup.fields.forEach(field => {
        const value = _get(groupValues, field.name);

        const blurError = getFieldError(value, field, formData, options);

        if (blurError) {
            _set(errors, [...groupPath, field.name].join('.'), {blurError});

            return;
        }

        const submitError = getFieldError(value, field, formData, {
            ...options,
            isSubmit: true,
        });

        if (submitError) {
            _set(errors, [...groupPath, field.name].join('.'), {submitError});
        }
    });
}

function validateFieldGroup(
    vGroup: IFieldGroup,
    formValues: IFormValues,
    errors: ValidationErrors,
    options: IValidationOptions,
): void {
    const isGroupValues =
        _has(formValues, vGroup.id) || options.isStrictStructure;
    const fGroupPath = isGroupValues ? [...vGroup.id.split('.')] : [];
    const fGroupValues = isGroupValues
        ? _get(formValues, vGroup.id)
        : formValues;

    const validationParams: IValidateFieldGroup = {
        formData: {
            formValues,
            groupValues: fGroupValues,
            groupPath: fGroupPath,
        },
        validationGroup: vGroup,
        errors,
        options,
    };

    if (Array.isArray(fGroupValues)) {
        fGroupValues.forEach((fArrayGroup, index) => {
            validationParams.formData.groupValues = fArrayGroup;
            validationParams.formData.groupPath = [...fGroupPath, index];

            validateGroupByValues(validationParams);
        });
    } else if (fGroupValues || !options.isStrictStructure) {
        validateGroupByValues(validationParams);
    }
}

function validateValues(
    validationInfo: IFormValidationInfo,
    formValues: IFormValues,
): ValidationErrors {
    const {fieldGroups, isStrictStructure} = validationInfo;
    const errors: ValidationErrors = {};
    const options: IValidationOptions = {
        isStrictStructure,
    };

    fieldGroups.forEach(vGroup => {
        validateFieldGroup(vGroup, formValues, errors, options);
    });

    return errors;
}

export default validateValues;
