const axios = require('axios');
const moment = require('moment');
const fs = require('fs-extra');
const path = require('path');
const chalk = require('chalk');

const {holidaysPath} = require('../webpack/utilities/getPathParams');

const RUS_COUNTRY_ID = 921;
const CALENDAR_API_HOST = 'https://api.calendar.yandex-team.ru';
const DATE_FORMAT = 'YYYY-MM-DD';
const HOLIDYAS_JSON_FILENAME = path.join(holidaysPath, 'holidays.json');

const calendarApi = axios.create({
    baseURL: CALENDAR_API_HOST,
});

downloadHolidaysToFs().then(
    () => console.info(chalk.green('Holidays succefully downloaded')),
    err => {
        console.error('Holidays download failure');
        console.error(err);
        process.exit(1);
    },
);

async function downloadHolidaysToFs() {
    const monthBelow = moment().subtract(1, 'month').format(DATE_FORMAT);

    const yearAfter = moment().add(1, 'year').format(DATE_FORMAT);

    await requestHolidays(monthBelow, yearAfter, RUS_COUNTRY_ID)
        .then(response => response.data)
        .then(processHolidaysResponse)
        .then(saveHolidaysMapToFS);
}

function requestHolidays(from, to, countryId) {
    return calendarApi.get('intapi/get-holidays', {
        params: {for: countryId, from, outMode: 'overrides', to},
    });
}

function processHolidaysResponse(calendarResponse) {
    return calendarResponse.holidays.reduce((acc, {date, type}) => {
        // Castyl: тикет на исправление https://st.yandex-team.ru/TRAVELFRONT-3596
        if (['2020-07-24', '2020-08-21', '2020-12-31'].includes(date)) {
            return acc;
        }

        acc[date] = type !== 'weekday';

        return acc;
    }, {});
}

function saveHolidaysMapToFS(holidaysMap) {
    if (fs.existsSync(HOLIDYAS_JSON_FILENAME)) {
        fs.unlinkSync(HOLIDYAS_JSON_FILENAME);
    }

    fs.ensureDirSync(holidaysPath);

    fs.writeFileSync(
        HOLIDYAS_JSON_FILENAME,
        JSON.stringify(holidaysMap, null, 4),
        'utf8',
    );
}
