import colors from 'colors/safe';
import moment, {Moment} from 'moment-timezone';
import {AxiosError} from 'axios';
import {flatten, groupBy, partition, shuffle} from 'lodash';

import {BUILD_TYPE_PR, TEAMCITY_DATE_FORMAT} from './constants/teamcity';

import {isNotNull} from '../../src/types/utilities';

import TeamcityApiClient, {
    TEAMCITY_DOMAIN,
} from '../utils/api/TeamcityApiClient/TeamcityApiClient';

interface ITestReport {
    browserId: string;
    fullTitle: string;
}

interface IRunInfo {
    date: Moment;
    buildId: number;
    failedCount: number;
    branchName: string;
}

interface IBuildTestReport extends ITestReport {
    runInfo: IRunInfo;
}

interface IFailedTestInfo {
    fullTitle: string;
    failedCount: number;
    failedPercentage: string;
    runsInfo: IRunInfo[];
    firstFail: Moment;
    lastFail: Moment;
}

// Названия тестов, которые не стоит учитывать в статистике
const IGNORE_TESTS: (string | RegExp)[] = [
    // /Записная книжка/,
    // /Рефреш формы траста/,
    // 'Портал Автовыбор саджеста в форме поиска при потере фокуса',
    // /Возврат брони после доплаты/,
    // /Бронь в рассрочку/,
    // /Портал/,
    // /Ошибка оплаты - /,
];

const HUMAN_DATE_FORMAT = 'DD.MM.YYYY';

const LINKS_LIMIT = 5;
const PRINT_ALL = true;
const PRINT_MOBILE = false;
const PRINT_DESKTOP = false;
const SORT_RUNS_BY = 'date' as 'date' | 'failedCount';

const groupTests = (
    failedTests: IBuildTestReport[],
    runCount: number,
): IFailedTestInfo[] => {
    return Object.entries(groupBy(failedTests, ({fullTitle}) => fullTitle))
        .map(([fullTitle, failedTests]) => ({
            fullTitle,
            failedCount: failedTests.length,
            failedPercentage: `${(
                (failedTests.length / runCount) *
                100
            ).toFixed(2)}%`,
            runsInfo: failedTests.map(({runInfo}) => runInfo),
            firstFail: failedTests[failedTests.length - 1].runInfo.date,
            lastFail: failedTests[0].runInfo.date,
        }))
        .sort(
            (testInfo1, testInfo2) =>
                testInfo2.failedCount - testInfo1.failedCount,
        );
};

const getReportLink = (buildId: number): string => {
    return `${TEAMCITY_DOMAIN}/repository/download/${BUILD_TYPE_PR}/${buildId}:id/html-report/index.html`;
};

const getTaskLink = (branchName: string): string | undefined => {
    const taskName = branchName.match(/^[A-Z]+-\d+/)?.[0];

    return taskName && `https://st.yandex-team.ru/${taskName}`;
};

const formatTestsToTable = (failedTestsGroups: IFailedTestInfo[]): string => {
    return failedTestsGroups
        .map(testInfo => {
            const tableValuePadding = ' '.repeat(18);
            const tableTestInfo = [
                {
                    name: 'Название',
                    value: colors.green(testInfo.fullTitle),
                },
                {
                    name: 'Упало раз',
                    value: colors.red(
                        `${testInfo.failedCount} (${testInfo.failedPercentage})`,
                    ),
                },
                {
                    name: 'Диапазон падений',
                    value: `${testInfo.firstFail.format(
                        HUMAN_DATE_FORMAT,
                    )} - ${testInfo.lastFail.format(HUMAN_DATE_FORMAT)}`,
                },
                {
                    name: 'Запуски',
                    value: shuffle(testInfo.runsInfo)
                        .slice(0, LINKS_LIMIT)
                        .sort((runInfo1, runInfo2) =>
                            SORT_RUNS_BY === 'failedCount'
                                ? runInfo1.failedCount - runInfo2.failedCount
                                : runInfo2.date.diff(runInfo1.date),
                        )
                        .map(runInfo => {
                            const taskLink = getTaskLink(runInfo.branchName);
                            const tableRunInfo = [
                                {
                                    name: 'Дата',
                                    value: runInfo.date.format(
                                        HUMAN_DATE_FORMAT,
                                    ),
                                },
                                {
                                    name: 'Всего упало',
                                    value: colors.red(
                                        String(runInfo.failedCount),
                                    ),
                                },
                                {
                                    name: 'Задача',
                                    value:
                                        taskLink ||
                                        colors.italic('Отсутствует'),
                                },
                                {
                                    name: 'Репорт',
                                    value: getReportLink(runInfo.buildId),
                                },
                            ];

                            return tableRunInfo
                                .map(
                                    ({name, value}) =>
                                        `${colors.bold(name)}: ${
                                            ' '.repeat(11 - name.length) + value
                                        }`,
                                )
                                .join('\n' + tableValuePadding);
                        })
                        .join('\n\n' + tableValuePadding),
                },
            ];

            return tableTestInfo
                .map(
                    ({name, value}) =>
                        `${colors.bold(name)}: ${
                            ' '.repeat(16 - name.length) + value
                        }`,
                )
                .join('\n');
        })
        .join('\n' + '-'.repeat(process.stdout.columns || 0) + '\n');
};

const isTestIgnored = (testName: string): boolean => {
    return IGNORE_TESTS.some(ignored =>
        typeof ignored === 'string'
            ? testName === ignored
            : ignored.test(testName),
    );
};

require('../utils/prepareEnvVars')();

(async (): Promise<void> => {
    const apiClient = new TeamcityApiClient();

    const startDate = moment().subtract(60, 'days'); // за последние n дней
    // const startDate = moment('2021-08-03'); // с конкретной даты

    const builds = await apiClient.getBuilds({
        buildType: BUILD_TYPE_PR,
        sinceDate: startDate.startOf('day').format(TEAMCITY_DATE_FORMAT),
        branch: {
            policy: 'ALL_BRANCHES',
        },
        count: 1e4,
        state: 'finished',
    });

    const reports: (IBuildTestReport[] | null)[] = await Promise.all(
        builds.map(async ({id: buildId, finishOnAgentDate, branchName}) => {
            let report: ITestReport[];

            try {
                report = await apiClient.getArtifactContent<ITestReport[]>(
                    buildId,
                    'reports/fallenTestsReport.json',
                );
            } catch {
                return null;
            }

            return report.map(testReport => ({
                ...testReport,
                runInfo: {
                    buildId,
                    date: moment(finishOnAgentDate, TEAMCITY_DATE_FORMAT),
                    failedCount: report.length,
                    branchName,
                },
            }));
        }),
    );

    const filteredReports = reports.filter(isNotNull);
    const runCount = filteredReports.length;
    const allFailedTests = flatten(filteredReports).filter(
        ({fullTitle}) => !isTestIgnored(fullTitle),
    );
    const [mobileFailedTests, desktopFailedTests] = partition(
        allFailedTests,
        ({browserId}) => browserId.includes('phone'),
    );

    console.log('Общее количество запусков тестов:', runCount);

    if (PRINT_ALL) {
        console.log();
        console.log('Статистика по всем тестам');
        console.log(
            formatTestsToTable(groupTests(allFailedTests, runCount * 2)),
        );
    }

    if (PRINT_MOBILE) {
        console.log();
        console.log('Статистика по мобильным тестам');
        console.log(
            formatTestsToTable(groupTests(mobileFailedTests, runCount)),
        );
    }

    if (PRINT_DESKTOP) {
        console.log();
        console.log('Статистика по десктопным тестам');
        console.log(
            formatTestsToTable(groupTests(desktopFailedTests, runCount)),
        );
    }
})().catch((err: AxiosError) => {
    if (err.response) {
        console.log(err.response);
    } else {
        console.log(err);
    }

    process.exit(1);
});
