import path from 'path';
import fs from 'fs-extra';
import moment from 'moment';

import {E2E_CLICKHOUSE_COLUMNS} from './constants/clickhouse';
import {BUILD_TYPE_RELEASE, TEAMCITY_DATE_FORMAT} from './constants/teamcity';
import {CLICKHOUSE_DATE_FORMAT} from '../utils/api/ClickHouseApiClient/constants';

import {IClickhouseReleaseRun} from './types/clickhouse';
import {THermioneTestsReport} from './types/hermione';

import ClickHouseApiClient from '../utils/api/ClickHouseApiClient/ClickHouseApiClient';
import TeamcityApiClient from '../utils/api/TeamcityApiClient/TeamcityApiClient';
import {getCurrentBranch} from '../utils/getCurrentBranch';
import {getAllFailedTriesCount} from './utils/getAllFailedTriesCount';

const DB_NAME = 'db_e2e_test_metrics';
const TABLE_NAME = 'e2e_release_runs';

const JSON_REPORT_PATH = './integration/json-report/report.json';

require('../utils/prepareEnvVars')();

(async (): Promise<void> => {
    const clickhouseUsername = process.env.CLICKHOUSE_E2E_USERNAME;
    const clickhousePassword = process.env.CLICKHOUSE_E2E_PASSWORD;

    if (!clickhouseUsername || !clickhousePassword) {
        throw new Error(
            'Требуются CLICKHOUSE_E2E_USERNAME и CLICKHOUSE_E2E_PASSWORD для авторизации в ClickHouse',
        );
    }

    const clickhouseClient = new ClickHouseApiClient<IClickhouseReleaseRun>({
        dbName: DB_NAME,
        tableName: TABLE_NAME,
        username: clickhouseUsername,
        password: clickhousePassword,
        columns: E2E_CLICKHOUSE_COLUMNS,
    });

    const teamcityClient = new TeamcityApiClient();

    const branchName = getCurrentBranch();
    const releaseTicketName = branchName.match(/^TRAVELFRONT-\d+/)?.[0];

    if (!releaseTicketName) {
        throw new Error(
            `Неверное название релизного тикета для ветки ${branchName}`,
        );
    }

    const builds = await teamcityClient.getBuilds({
        buildType: BUILD_TYPE_RELEASE,
        count: 1e4,
        state: 'running',
        branch: {
            policy: 'ALL_BRANCHES',
        },
    });

    const branchBuild = builds.find(
        ({branchName: buildBranchName}) =>
            buildBranchName === releaseTicketName,
    );

    if (!branchBuild) {
        throw new Error('Не найден запущенный билд релизных тестов');
    }

    const detailedBuild = await teamcityClient.getBuild(branchBuild.id);
    const jsonReport: THermioneTestsReport = await fs.readJson(
        path.resolve(JSON_REPORT_PATH),
    );

    const duration =
        Date.now() -
        Number(moment(detailedBuild.startDate, TEAMCITY_DATE_FORMAT));
    const passedCount = detailedBuild.testOccurrences?.passed ?? 0;
    const failedCount = detailedBuild.testOccurrences?.failed ?? 0;
    const ignoredCount = detailedBuild.testOccurrences?.ignored ?? 0;
    const retriesCount = getAllFailedTriesCount(jsonReport) - failedCount;

    const whereQuery = `ticketName='${releaseTicketName}'`;
    const [currentReleaseRun] = await clickhouseClient.getValues(whereQuery);

    const runValues = {
        date: moment().utc().format(CLICKHOUSE_DATE_FORMAT),
        passedCount,
        failedCount,
        ignoredCount,
        retriesCount,
        duration,
    };

    if (currentReleaseRun) {
        await clickhouseClient.updateValue(runValues, whereQuery);
    } else {
        await clickhouseClient.insertValue({
            ticketName: releaseTicketName,
            ...runValues,
        });
    }
})().catch(err => {
    console.log(err);

    process.exit(1);
});
