import moment from 'moment';

import {
    LIGHTHOUSE_RELEASE_METRICS_TABLE_COLUMNS,
    RELEASE_PAGES_URLS,
} from './constants';
import {CLICKHOUSE_DATE_FORMAT} from '../../utils/api/ClickHouseApiClient/constants';

import {EReleasePage, IClickHouseLighthouseMeasure} from './types';
import {ILighthouseScoreAndMetrics} from '../types';

import getTestingVersion from './utilities/getTestingVersion';
import ClickHouseApiClient from '../../utils/api/ClickHouseApiClient/ClickHouseApiClient';
import getReleasePagesMetrics from './utilities/getReleasePagesMetrics/getReleasePagesMetrics';
import hasAlreadyMeasured from './utilities/hasAlreadyMeasured';

const DB_NAME = 'db_lighthouse';
const TABLE_NAME = 'releaseMetrics';

(async (): Promise<void> => {
    try {
        if (
            !process.env.CLICKHOUSE_LIGHTHOUSE_USERNAME ||
            !process.env.CLICKHOUSE_LIGHTHOUSE_PASSWORD
        ) {
            throw new Error('Отсутствуют данные для авторизации в clickhouse');
        }

        if (
            !process.env.ITERATIONS_PER_PAGE ||
            !process.env.CONCURRENT_MEASURES
        ) {
            throw new Error('Отсутствуют опции для запуска замеров');
        }

        const lighthouseClickHouseClient =
            new ClickHouseApiClient<IClickHouseLighthouseMeasure>({
                dbName: DB_NAME,
                tableName: TABLE_NAME,
                username: process.env.CLICKHOUSE_LIGHTHOUSE_USERNAME,
                password: process.env.CLICKHOUSE_LIGHTHOUSE_PASSWORD,
                columns: LIGHTHOUSE_RELEASE_METRICS_TABLE_COLUMNS,
            });

        const testingVersion = await getTestingVersion();

        if (
            await hasAlreadyMeasured(testingVersion, lighthouseClickHouseClient)
        ) {
            console.log('Lighthouse: эта версия тестинга уже замерялась');

            return;
        }

        const startMeasuringTime = moment();

        console.log('Lighthouse: запуск замеров');

        const pagesMetrics = await getReleasePagesMetrics(
            Number(process.env.CONCURRENT_MEASURES),
            Number(process.env.ITERATIONS_PER_PAGE),
        );

        console.log(
            `Lighthouse: замеры заняли минут: ${moment().diff(
                startMeasuringTime,
                'minutes',
            )} (страниц: ${
                Object.keys(RELEASE_PAGES_URLS).length
            }, замеров на страницу: ${Number(process.env.ITERATIONS_PER_PAGE)},
            параллельных потоков: ${Number(process.env.CONCURRENT_MEASURES)})`,
        );

        const dateUtc = moment().utc().format(CLICKHOUSE_DATE_FORMAT);

        const metrics = (
            Object.entries(pagesMetrics) as [
                EReleasePage,
                ILighthouseScoreAndMetrics,
            ][]
        ).map(([pageId, pageMetrics]) => ({
            date: dateUtc,
            platform: 'mobile',
            version: testingVersion,
            page: pageId,
            score: Math.round(pageMetrics.score),
            fcp: Math.round(pageMetrics.fcp),
            lcp: Math.round(pageMetrics.lcp),
            si: Math.round(pageMetrics.speedIndex),
            tti: Math.round(pageMetrics.tti),
            tbt: Math.round(pageMetrics.tbt),
            cls: Math.floor(pageMetrics.cls * 100),
            mpf: Math.round(pageMetrics.maxPotentialFid),
            ttfb: Math.round(pageMetrics.ttfb),
            mtw: Math.round(pageMetrics.mainThreadWork),
            jset: Math.round(pageMetrics.jsExecutionTime),
            ds: Math.round(pageMetrics.domSize),
        }));

        if (
            await hasAlreadyMeasured(testingVersion, lighthouseClickHouseClient)
        ) {
            console.log('Lighthouse: эта версия тестинга уже замерялась');

            return;
        }

        await lighthouseClickHouseClient.insertValues(metrics);

        console.log('Lighthouse: замеры успешно загружены');
    } catch (err) {
        console.error(
            'Lighthouse: ошибка подсчета и отправки lighthouse метрик релиза',
            err,
        );
    }
})();
