import micromatch from 'micromatch';

import {TDiffEntry} from '../../utils/api/GithubApiClient/types/pulls';

import GithubApiClient from '../../utils/api/GithubApiClient/GithubApiClient';
import {getCurrentBranch} from '../../utils/getCurrentBranch';

export enum ESkipType {
    LINT = 'lint',
    E2E = 'e2e',
    UNIT_TESTS = 'unitTests',
}

export interface IVerifySkipResult {
    canBeSkipped: boolean;
    noSkipFiles: TDiffEntry[];
}

const SKIP_PATTERNS: Record<ESkipType, string[]> = {
    [ESkipType.LINT]: ['(*.testpalm.yaml|*.md)'],
    [ESkipType.E2E]: ['(*.testpalm.yaml|*.md)'],
    [ESkipType.UNIT_TESTS]: ['(*.testpalm.yaml|*.md)'],
};

/**
 * Утилита, которая помогает понять, можно ли скипнуть какую-то проверку на PR
 * Если все файлы удовлетворяют какому-то скип-паттерну, тогда проверку можно скипнуть
 */
export async function verifySkip(
    skipType: ESkipType,
): Promise<IVerifySkipResult> {
    const patterns = SKIP_PATTERNS[skipType];
    const apiClient = new GithubApiClient();

    const currentBranch = getCurrentBranch();
    const [pullRequest] = await apiClient.getPullRequests({
        state: 'open',
        branchName: currentBranch,
    });

    if (!pullRequest) {
        throw new Error('No pull request found');
    }

    const files = await apiClient.getPullRequestFiles(pullRequest.number);

    const noSkipFiles = files.filter(({filename}) => {
        return patterns.every(pattern => {
            return !micromatch.isMatch(filename, pattern, {
                dot: true,
                matchBase: !pattern.includes('/'),
            });
        });
    });

    return {
        canBeSkipped: noSkipFiles.length === 0,
        noSkipFiles,
    };
}
