import {resolve} from 'path';
import {pickBy, mapValues} from 'lodash';
import getFunctionName from '@yandex-data-ui/i18n-ts/client/getFunctionName';
import fse from 'fs-extra';

import {srcPath, rootPath} from '../../../webpack/utilities/getPathParams';
import {processFiles} from './utilities/processFiles/processFiles';
import getTankerUrl from './utilities/getTankerUrl';
import renderKeySetListHtml from './utilities/renderKeySetListHtml';
import getTmpDirPath from './utilities/getTmpDirPath';

import {readFromFiles} from '../readFromFilesAndBuild/readFromFiles';

const OUTPUT = resolve(getTmpDirPath(), 'unused-tanker-report.html');

const folders = [srcPath, resolve(rootPath, 'server')];
const usedKeysets = processFiles(folders);

(async (): Promise<void> => {
    const translations = await readFromFiles();

    translations.keySets = pickBy(
        translations.keySets,
        keySet => !usedKeysets.unsafeUsages[keySet.name],
    );

    const fullUnusedKeySetsName: string[] = [];

    translations.keySets = pickBy(translations.keySets, keySet => {
        const usages = usedKeysets.preciseUsages[keySet.name];

        if (!usages) {
            fullUnusedKeySetsName.push(keySet.name);
        }

        return Boolean(usages);
    });

    let unusedKeys = mapValues(translations.keySets, keySet => {
        const keySetUsages = usedKeysets.preciseUsages[keySet.name];

        const unused = pickBy(keySet.keys, key => {
            return !keySetUsages[getFunctionName(key.name)];
        });

        if (!Object.keys(unused).length) {
            return {};
        }

        return mapValues(unused, key => getTankerUrl(keySet.name, key.name));
    });

    unusedKeys = pickBy(unusedKeys, keySet => Object.keys(keySet).length);

    await fse.ensureDir(getTmpDirPath());

    await fse.writeFile(
        OUTPUT,
        renderKeySetListHtml(unusedKeys, fullUnusedKeySetsName, {
            title: 'Unused Tanker keys',
            description: 'Ключи которые не используются в коде',
        }),
    );

    console.log(`report: file://${OUTPUT}`);
})().catch(e => {
    console.error(e);

    process.exit(1);
});
