import fs from 'fs';
import {resolve} from 'path';
import merge from 'lodash/merge';
import {parse} from '@babel/parser';

import findUsages, {IFindUsagesResult} from './utilities/findUsages';

import isDirectory from '../isDirectory';
import isScript from '../isScript';

/**
 * Находит использования ключей в танкере
 * @param folders - массив с путями до папок, в которых анализируются файлы
 * @return найденные ключи в объекте вида {[keysetName: string]: {[key: string]: true}}
 */
export function processFiles(folders: string[]): IFindUsagesResult {
    const result: IFindUsagesResult = {
        preciseUsages: {},
        unsafeUsages: {},
    };

    function walkSync(dirPath: string): void {
        const allFiles = fs.readdirSync(dirPath);
        const directories = allFiles
            .map(file => resolve(dirPath, file))
            .filter(isDirectory);
        const filesWithCode = allFiles.filter(isScript);

        filesWithCode.forEach(function (file) {
            const fullPath = resolve(dirPath, file);

            const code = fs.readFileSync(fullPath, 'utf8');

            const ast = parse(code, {
                plugins: [
                    'jsx',
                    'typescript',
                    'dynamicImport',
                    'classProperties',
                    'decorators-legacy',
                    'optionalChaining',
                    'nullishCoalescingOperator',
                ],
                sourceType: 'unambiguous',
            });

            const usages = findUsages(ast);

            merge(result, usages);
        });

        directories.forEach(function (dirName) {
            const fullPath = resolve(dirPath, dirName);

            walkSync(fullPath);
        });
    }

    folders.forEach(folder => {
        walkSync(folder);
    });

    return result;
}
