import axios, {AxiosInstance} from 'axios';
import https from 'https';

export type TClickHouseColumns<Data> = {
    [Column in keyof Data]: 'DateTime' | 'String' | 'Int8' | 'Int32';
};

interface IClickHouseConfig<Data> {
    dbName: string;
    tableName: string;
    username: string;
    password: string;
    columns: TClickHouseColumns<Data>;
}

interface IClickHouseValues<Data> {
    meta: {name: string; type: string}[];
    data: Data[];
    rows: number;
    statistics: unknown;
}

type TUpdateValues<Data, UpdateKeys extends keyof Data> = {
    [Key in UpdateKeys]: Data[Key];
};

const CLICKHOUSE_URL = 'https://man-iajiyg2g16081u8t.db.yandex.net:8443';

export default class ClickHouseApiClient<
    Data extends {[Key in keyof Data]: string | number},
> {
    private readonly axiosInstance: AxiosInstance = axios.create();
    private readonly dbTableName: string;
    private readonly columns: TClickHouseColumns<Data>;
    private readonly columnNames: (keyof Data)[];

    constructor(config: IClickHouseConfig<Data>) {
        this.axiosInstance = axios.create({
            baseURL: CLICKHOUSE_URL,
            headers: {
                'X-ClickHouse-User': config.username,
                'X-ClickHouse-Key': config.password,
                accept: 'application/json',
            },
            httpsAgent: new https.Agent({
                rejectUnauthorized: false,
            }),
        });
        this.dbTableName = `${config.dbName}.${config.tableName}`;
        this.columns = config.columns;
        this.columnNames = Object.keys(config.columns) as (keyof Data)[];
    }

    async getValues(where?: string): Promise<Data[]> {
        const response = await this.axiosInstance.get<IClickHouseValues<Data>>(
            '/',
            {
                params: {
                    query: `
                  SELECT * FROM ${this.dbTableName}
                  ${where ? `WHERE ${where}` : ''}
                  FORMAT JSON
                `,
                },
            },
        );

        return response.data.data;
    }

    async insertValue(value: Data): Promise<void> {
        return this.insertValues([value]);
    }

    async insertValues(values: Data[]): Promise<void> {
        const stringifyColumnNames = this.columnNames.join(', ');

        const stringifyValues = values
            .map(itemValues => `(${this.stringifyValues(itemValues)})`)
            .join(', ');

        await this.axiosInstance.post(
            '/',
            `
            INSERT INTO ${this.dbTableName}
            (${stringifyColumnNames}) VALUES ${stringifyValues}`,
        );
    }

    async deleteValues(where?: string): Promise<void> {
        await this.axiosInstance.post(
            '/',
            `
            ALTER TABLE ${this.dbTableName} DELETE WHERE ${where || '1=1'}`,
        );
    }

    async updateValue<UpdateKeys extends keyof Data>(
        itemValues: TUpdateValues<Data, UpdateKeys>,
        where: string,
    ): Promise<void> {
        const stringifyUpdate = Object.entries(itemValues)
            .map(([column, value]) => `${column}=${this.stringifyValue(value)}`)
            .join(', ');

        await this.axiosInstance.post(
            '/',
            `
            ALTER TABLE ${this.dbTableName}
            UPDATE ${stringifyUpdate}
            WHERE ${where}`,
        );
    }

    async createTableIfNotExist(orderByColumn: keyof Data): Promise<void> {
        const stringifyColumns = Object.entries(this.columns)
            .map(([name, type]) => `${name} ${type}`)
            .join(', ');

        await this.axiosInstance.post(
            '/',
            `
            CREATE TABLE IF NOT EXISTS ${this.dbTableName}
            (${stringifyColumns})
            ENGINE = MergeTree()
            ORDER BY ${orderByColumn}
            FORMAT JSON`,
        );
    }

    async deleteTable(): Promise<void> {
        await this.axiosInstance.post(
            '/',
            `
            DROP TABLE IF EXISTS ${this.dbTableName}`,
        );
    }

    private stringifyValue(value: unknown): string {
        return typeof value === 'string' ? `'${value}'` : `${value}`;
    }

    private stringifyValues(itemValues: Data): string {
        return this.columnNames
            .map(column => this.stringifyValue(itemValues[column]))
            .join(', ');
    }
}
