import {Octokit} from '@octokit/rest';

import {TDiffEntry, TPullRequest, TPullRequestSimple} from './types/pulls';

export interface IGetPullsOptions {
    state?: 'open' | 'closed' | 'all';
    branchName?: string;
}

class GithubApiClient {
    // GITHUB_OAUTH_KEY - наш
    // GITHUB_TOKEN     - в teamcity
    private readonly githubToken = process.env.GITHUB_TOKEN;
    private readonly owner = 'travel-ui';
    private readonly repo = 'ya-travel';
    private readonly client = new Octokit({
        baseUrl: 'https://github.yandex-team.ru/api/v3',
        auth: this.githubToken,
    });

    async getPullRequests(
        options: IGetPullsOptions = {},
    ): Promise<TPullRequestSimple[]> {
        const pulls: TPullRequestSimple[] = [];
        const pageSize = 100;
        let page = 1;

        // eslint-disable-next-line no-constant-condition
        while (true) {
            const {data: pagePulls} = await this.client.pulls.list({
                owner: this.owner,
                repo: this.repo,
                head:
                    options.branchName && `${this.owner}:${options.branchName}`,
                page: page++,
                per_page: pageSize,
                state: options.state,
            });

            pulls.push(...pagePulls);

            if (pagePulls.length < pageSize) {
                break;
            }
        }

        return pulls;
    }

    async getPullRequest(pullNumber: number): Promise<TPullRequest> {
        const {data} = await this.client.pulls.get({
            owner: this.owner,
            repo: this.repo,
            pull_number: pullNumber,
        });

        return data;
    }

    async getPullRequestFiles(pullNumber: number): Promise<TDiffEntry[]> {
        const pageSize = 100;
        const files: TDiffEntry[] = [];
        let page = 1;

        // eslint-disable-next-line no-constant-condition
        while (true) {
            const {data: pageFiles} = await this.client.pulls.listFiles({
                owner: this.owner,
                repo: this.repo,
                pull_number: pullNumber,
                page: page++,
                per_page: pageSize,
            });

            files.push(...pageFiles);

            if (pageFiles.length < pageSize) {
                break;
            }
        }

        return files;
    }
}

export default GithubApiClient;
