import axios from 'axios';

import {
    IBuildCancelRequest,
    IRunBuildParams,
    ITeamcityBuildLong,
    ITeamcityBuildShort,
    ITeamcityGetBuildsResponse,
} from './types/builds';

export interface IKeyValueLocatorValues {
    [key: string]: string | number | boolean | IKeyValueLocatorValues;
}

export type TLocatorValues = IKeyValueLocatorValues | IKeyValueLocatorValues[];

export const TEAMCITY_DOMAIN = 'https://teamcity.yandex-team.ru';

class TeamcityApiClient {
    /**
     * https://www.jetbrains.com/help/teamcity/rest/teamcity-rest-api-documentation.html#Locator
     *
     * Локатор в конечном виде - строка вида key1:value,key2:value2,...
     * Сюда перадется объект, который строит эту строку
     * Могут быть вложенные; также можно передать массив локаторов,
     * тогда результат будет объединением результатов этих локаторов
     */
    static buildLocator(values: TLocatorValues): string {
        if (Array.isArray(values)) {
            return values
                .map(
                    locatorValues =>
                        `item:(${TeamcityApiClient.buildLocator(
                            locatorValues,
                        )})`,
                )
                .join(',');
        }

        return Object.entries(values)
            .map(([key, value]) => {
                const valueString =
                    typeof value === 'object'
                        ? `(${TeamcityApiClient.buildLocator(value)})`
                        : value;

                return `${key}:${valueString}`;
            })
            .join(',');
    }

    // TEAMCITY_OAUTH_KEY - наш
    // TEAMCITY_TOKEN     - в teamcity
    // TS_TOKEN           - в ферме
    private readonly teamcityToken =
        process.env.TEAMCITY_OAUTH_KEY ||
        process.env.TEAMCITY_TOKEN ||
        process.env.TS_TOKEN;
    private readonly axiosInstance = axios.create({
        baseURL: TEAMCITY_DOMAIN,
        headers: {
            authorization: `OAuth ${this.teamcityToken}`,
            accept: 'application/json',
        },
    });

    // https://www.jetbrains.com/help/teamcity/rest/buildapi.html#cancelMultiple
    async cancelBuilds(
        buildIds: number[],
        cancelRequest: IBuildCancelRequest = {},
    ): Promise<void> {
        const buildIdLocator = TeamcityApiClient.buildLocator(
            buildIds.map(id => ({id})),
        );

        await this.axiosInstance.post(
            `/app/rest/builds/multiple/${buildIdLocator}`,
            cancelRequest,
        );
    }

    // https://www.jetbrains.com/help/teamcity/rest/buildapi.html#getBuild
    async getBuild(buildId: number): Promise<ITeamcityBuildLong> {
        const buildIdLocator = TeamcityApiClient.buildLocator({id: buildId});

        const {data} = await this.axiosInstance.get<ITeamcityBuildLong>(
            `/app/rest/builds/${buildIdLocator}`,
        );

        return data;
    }

    // https://www.jetbrains.com/help/teamcity/rest/buildapi.html#getAllBuilds
    async getBuilds(
        locatorValues: TLocatorValues,
    ): Promise<ITeamcityBuildShort[]> {
        const {data} = await this.axiosInstance.get<ITeamcityGetBuildsResponse>(
            '/app/rest/builds',
            {
                params: {
                    locator: TeamcityApiClient.buildLocator(locatorValues),
                },
            },
        );

        return data.build;
    }

    async getArtifactContent<T>(buildId: number, path: string): Promise<T> {
        const {data} = await this.axiosInstance.get<T>(
            `/app/rest/builds/${buildId}/artifacts/content/${path}`,
        );

        return data;
    }

    async runBuild(params: IRunBuildParams): Promise<ITeamcityBuildShort> {
        const {data} = await this.axiosInstance.post<ITeamcityBuildShort>(
            `/app/rest/buildQueue`,
            params,
        );

        return data;
    }
}

export default TeamcityApiClient;
