const axios = require('axios').default;

/**
 * @typedef VaultVersionResponse
 * @prop {'ok' | 'error' | 'warning'} status
 * @prop {VaultSecretVersion} version
 */

/**
 * @typedef VaultSecretVersion
 * @prop {string} comment
 * @prop {number} created_at
 * @prop {number} created_by
 * @prop {string} creator_login
 * @prop {string} secret_name
 * @prop {string} secret_uuid
 * @prop {VaultVersionValue[]} value
 * @prop {string} version
 */

/**
 * @typedef VaultVersionValue
 * @prop {string} key
 * @prop {string} value
 */

const YAV_ENDPOINT = 'https://vault-api.passport.yandex.net/1';

const VAULT_AUTH_KEY = process.env.VAULT_AUTH_KEY;

if (!VAULT_AUTH_KEY) {
    console.error('VAULT_AUTH_KEY must be specified in env variables');
    process.exit(1);
}

const yavClient = axios.create({
    baseURL: YAV_ENDPOINT,
    headers: {
        Authorization: VAULT_AUTH_KEY,
    },
});

/**
 * @param {string} version
 *
 * @returns {Promise<VaultVersionResponse>}
 */
function getSecretByVersion(version) {
    return yavClient.get(`/versions/${version}/`).then(res => res.data);
}

/**
 * @param {string} version
 *
 * @returns {Promise<void>}
 */
function setEnvVarsWithSecret(version) {
    return getSecretByVersion(version).then(response => {
        if (response.status !== 'ok') {
            return;
        }
        response.version.value.forEach(secret => {
            process.env[secret.key] = secret.value;
        });
    });
}

exports.setEnvVarsWithSecret = setEnvVarsWithSecret;

/**
 * @param {string} version
 *
 * @returns {Promise<string>}
 */
function getEnvVarsStringWithSecret(version) {
    return getSecretByVersion(version).then(response => {
        if (response.status !== 'ok') {
            return '';
        }
        return response.version.value
            .map(({key, value}) => `${key}=${value}`)
            .join('\n');
    });
}

exports.getEnvVarsStringWithSecret = getEnvVarsStringWithSecret;
