const path = require('path');
const webpack = require('webpack');
const chalk = require('chalk');
const MomentLocalesPlugin = require('moment-locales-webpack-plugin');

const {
    srcPath,
    rootPath,
    i18nPath,
    excludeNodeModulesRegExpPath,
    nodeModulesPath,
    stylesPath,
    freezePath,
} = require('./utilities/getPathParams');
const getGlobalVariables = require('./utilities/getGlobalVariables');
const getStaticFileNameSuffix = require('./utilities/getStaticFileNameSuffix');
const getBuildReplacementProjectsPlugin = require('./utilities/getBuildReplacementProjectsPlugin');

const {
    nodeEnv,
    buildLang,
    isDevelopment,
    developmentBuildProjects,
    isThemekitDebug,
    consoleReduxDevTools,
} = require('./variables');

const IMAGES_REG_EX = /\.(ico|bmp|gif|jpe?g|png|svg|webp)$/;

/* Styles Loaders */
const scssLoaders = [
    {
        loader: 'resolve-url-loader',
        options: {
            sourceMap: true,
        },
    },
    {
        loader: 'postcss-loader',
        options: {
            sourceMap: true,
        },
    },
    {
        loader: 'sass-loader',
        options: {
            implementation: require.resolve('sass'),
            sourceMap: true,
            sassOptions: {
                includePaths: [stylesPath],
            },
        },
    },
    {
        loader: 'sass-resources-loader',
        options: {
            sourceMap: true,
            hoistUseStatements: true,
            resources: [
                path.join(stylesPath, 'mixins.scss'),
                path.join(stylesPath, 'variables.scss'),
                path.join(stylesPath, 'spacing.scss'),
            ],
        },
    },
];

const cssLoaders = [];

const staticFileNameSuffix = getStaticFileNameSuffix();

module.exports = ({isServer}) => {
    const commonWebpackPlugins = [
        new webpack.DefinePlugin({
            ...getGlobalVariables(isServer),
            'process.env.NODE_ENV': JSON.stringify(nodeEnv),
            'process.env.BEM_LANG': JSON.stringify(buildLang),
            'process.env.DEBUG_LEGO_THEMEKIT': JSON.stringify(isThemekitDebug),
            'process.env.TRAVEL_CONSOLE_REDUX_DEV_TOOLS':
                JSON.stringify(consoleReduxDevTools),
        }),
        new MomentLocalesPlugin({localesToKeep: ['ru']}),
    ];

    if (isDevelopment && developmentBuildProjects) {
        if (!isServer) {
            console.log(
                chalk.green(`Build projects: ${developmentBuildProjects}`),
            );
        }

        commonWebpackPlugins.unshift(getBuildReplacementProjectsPlugin());
    }

    return {
        mode: isDevelopment ? 'development' : 'production',
        context: srcPath,
        module: {
            rules: [
                {
                    test: /\.scss$/,
                    exclude: excludeNodeModulesRegExpPath,
                    use: scssLoaders,
                },
                {
                    test: /\.css$/,
                    exclude: excludeNodeModulesRegExpPath,
                    use: cssLoaders,
                },
                {
                    test: IMAGES_REG_EX,
                    include: /node_modules\/@yandex-lego/,
                    type: 'asset/resource',
                    generator: {
                        publicPath: freezePath,
                        filename: `[contenthash][ext]${staticFileNameSuffix}`,
                    },
                },
                {
                    test: IMAGES_REG_EX,
                    exclude: /node_modules/,
                    type: 'asset/resource',
                    generator: {
                        publicPath: freezePath,
                        filename: `[contenthash][ext]${staticFileNameSuffix}`,
                    },
                },
            ],
        },
        plugins: commonWebpackPlugins,
        resolve: {
            fallback: {
                path: false,
                constants: false,
                crypto: false,
                zlib: false,
                stream: false,
                https: false,
                http: false,
                fs: false,
                vm: false,
                os: false,
            },
            alias: {
                src: srcPath,
                sagas: path.join(srcPath, 'redux', 'sagas'),
                reducers: path.join(srcPath, 'redux', 'reducers'),
                selectors: path.join(srcPath, 'redux', 'selectors'),
                components: path.join(srcPath, 'components'),
                polyfills: path.join(srcPath, 'polyfills'),
                projects: path.join(srcPath, 'projects'),
                icons: path.join(srcPath, 'icons'),
                containers: path.join(srcPath, 'containers'),
                utilities: path.join(srcPath, 'utilities'),
                constants: path.join(srcPath, 'constants'),
                contexts: path.join(srcPath, 'contexts'),
                hooks: path.join(srcPath, 'hooks'),
                ui: path.join(srcPath, 'ui'),
                i18nNew: path.join(i18nPath, 'keysets'),
                types: path.join(srcPath, 'types'),
                server: path.join(rootPath, 'server'),
                serviceProvider: path.join(srcPath, 'serviceProvider'),
                'moment-timezone': path.join(
                    nodeModulesPath,
                    'moment-timezone',
                    'builds',
                    'moment-timezone-with-data-2012-2022.min',
                ),
                images: path.join(srcPath, 'images'),
            },
            extensions: ['.json', '.ts', '.tsx', '.jsx', '.js', '.scss'],
        },
        resolveLoader: {
            modules: [
                'node_modules',
                path.join(rootPath, 'webpack', 'loaders'),
            ],
        },
    };
};
