import path from 'path';
import fs from 'fs-extra';
import {PluginItem, TransformOptions} from '@babel/core';
import {RuleSetRule} from 'webpack';

import {EWebpackTarget} from 'types/EWebpackTarget';

import {excludeNodeModulesRegExpPath} from './utilities/getPathParams';

import {patchOptionsPresetAndPlugins} from './babelUtils';
import {isDevelopment, withHotReload} from './variables';

const babelrcPath = path.join(__dirname, '../.babelrc');
const babelrc: TransformOptions = fs.readJsonSync(babelrcPath);

function getPresets(target: EWebpackTarget): PluginItem[] {
    const presets = [...(babelrc.presets || [])];

    if (target === EWebpackTarget.NODE) {
        return presets;
    }

    return patchOptionsPresetAndPlugins(presets, '@babel/preset-env', {
        modules: false,
        corejs: '3.19',
        useBuiltIns: 'entry',
    });
}

function getPlugins(target: EWebpackTarget): PluginItem[] {
    let plugins = [...(babelrc.plugins || [])];

    if (target === EWebpackTarget.WEB) {
        plugins = plugins
            // Меняем динамические импорты для клиентской части
            .filter(plugin => plugin !== 'babel-plugin-dynamic-import-node')
            .concat('@babel/plugin-syntax-dynamic-import');

        if (withHotReload) {
            plugins = plugins.concat('react-hot-loader/babel');
        }
    }

    return plugins;
}

export function getJsRules(target: EWebpackTarget): RuleSetRule[] {
    const presets = getPresets(target);
    const plugins = getPlugins(target);

    const forServer = target === EWebpackTarget.NODE;
    const cacheDirectory = isDevelopment
        ? `node_modules/.cache/babel/${forServer ? 'server' : 'client'}`
        : false;
    const sourceType = forServer ? undefined : 'unambiguous';

    return [
        {
            // Обработка файлов проекта
            test: /\.([tj])sx?$/,
            exclude: /node_modules/,
            use: [
                {
                    loader: 'babel-loader',
                    options: {
                        cacheDirectory,
                        cacheCompression: false,
                        babelrc: false,
                        sourceType,
                        presets,
                        plugins,
                    },
                },
            ],
        },
        {
            // Обработка node_modules
            test: /node_modules\/.*\.([tj])sx?$/,
            exclude: forServer
                ? excludeNodeModulesRegExpPath
                : [/core-js/, /webpack/],
            use: [
                {
                    loader: 'babel-loader',
                    options: {
                        cacheDirectory,
                        cacheCompression: false,
                        babelrc: false,
                        sourceType,
                        presets,
                        plugins: [],
                    },
                },
            ],
        },
    ];
}
