import { TOptionalLabel, TLabel } from "../Labels/ILabel";
import { Gauge } from "../Sensors/Gauge";
import { Counter } from "../Sensors/Counter";
import { Rate } from "../Sensors/Rate";
import { TSensors } from "../Sensors/types/TSensors";
import { ISingleValueSensor } from "../Sensors/types/ISingleValueSensor";
import { ISerialValueSensor } from "../Sensors/types/ISerialValueSensor";
import { timestamp } from "../utils/timestamp";

interface ISensorData {
  ts: number;
  sensors: Array<ISingleValueSensor | ISerialValueSensor>;
}

export class Registry {
  private readonly registry: Map<TLabel, TSensors> = new Map();

  private getCommonLabels = (name: string, labels: TOptionalLabel): TLabel => {
    return {
      sensor: name,
      ...labels,
    };
  };

  private findSensorByLabel = (label: TLabel): TSensors | null => {
    for (const [registryLabel, sensor] of this.registry) {
      if (registryLabel === label) {
        return sensor;
      }

      if (this.isSameLabels(registryLabel, label)) {
        return sensor;
      }
    }

    return null;
  };

  private isSameLabels = (
    registryLabel: TLabel,
    sensorLabel: TLabel
  ): boolean => {
    if (registryLabel.sensor === sensorLabel.sensor) {
      for (const key in registryLabel) {
        if (registryLabel[key] !== sensorLabel[key]) {
          return false;
        }
      }

      return (
        Object.keys(registryLabel).length === Object.keys(sensorLabel).length
      );
    }

    return false;
  };

  private getSensor = <T extends TSensors>(
    name: string,
    labels: TOptionalLabel,
    expectedSensor: new (labels: TLabel) => T
  ) => {
    const label = this.getCommonLabels(name, labels);
    let sensor = this.findSensorByLabel(label) as T | null;

    if (sensor) {
      return sensor;
    }

    sensor = new expectedSensor(label);
    this.registry.set(label, sensor);
    return sensor;
  };

  public gauge = (name: string, labels: TOptionalLabel): Gauge => {
    return this.getSensor(name, labels, Gauge);
  };

  public counter = (name: string, labels: TOptionalLabel): Counter => {
    return this.getSensor(name, labels, Counter);
  };

  public rate = (name: string, labels: TOptionalLabel): Rate => {
    return this.getSensor(name, labels, Rate);
  };

  public getData = (): ISensorData => {
    const data: ISensorData = {
      ts: timestamp(),
      sensors: [],
    };

    for (const [, sensor] of this.registry) {
      if (sensor.hasValues()) {
        data.sensors.push(sensor.serialize());
        sensor.reset();
      }
    }

    return data;
  };
}
