import {
    ISearchPageIdentifyParams,
    TFilterSlug,
} from '../../hotels/seo-pages/city/models';
import {EPlainTextStyle} from './EPlainTextStyle';
import {THotelSlug, TGeoRegionSlug} from '../../hotels/common/common';
import {IPrice} from '../common';
import {
    IRouteLinkBlock,
    ISearchAviaLinkBlock,
} from '../../avia/seo-pages/common/ISectionTextBlock';

export enum ESectionBlock {
    SECTION_TEXT_BLOCK = 'ISectionTextBlock',
    SUB_SECTION_TEXT_BLOCK = 'ISubSectionTextBlock',
    TEXT_BLOCK = 'ITextBlock',
    SPOILER_BLOCK = 'ISpoilerBlock',
    PRICE_TEXT_BLOCK = 'IPriceTextBlock',
    PLAIN_TEXT_BLOCK = 'IPlainTextBlock',
    EXTERNAL_LINK_BLOCK = 'IExternalLinkBlock',
    HOTEL_LINK_BLOCK = 'IHotelLinkBlock',
    REGION_LINK_BLOCK = 'IRegionLinkBlock',
    SEARCH_HOTELS_LINK_BLOCK = 'ISearchHotelsLinkBlock',
    GEO_LINK_GROUP_BLOCK = 'IGeoLinkGroupBlock',
}

/**
 * Внешняя ссылка, url приходит уже с бекенда
 */
export interface IExternalLinkBlock {
    type: ESectionBlock.EXTERNAL_LINK_BLOCK;
    data: {
        /**
         * Текст ссылки
         */
        text: string;
        /**
         * Ссылка
         */
        url: string;
    };
}

/**
 * Ссылка на страницу отеля, саму ссылку формирует фронт
 */
export interface IHotelLinkBlock {
    type: ESectionBlock.HOTEL_LINK_BLOCK;
    data: {
        /**
         * Текст ссылки
         */
        text: string;
        /**
         * Параметры для формирования ссылки
         */
        urlParams: {
            hotelSlug: THotelSlug;
        };
    };
}

export interface IRegionLinkBlock {
    type: ESectionBlock.REGION_LINK_BLOCK;
    data: {
        /**
         * Текст ссылки
         */
        text: string;
        /**
         * Параметры для формирования ссылки
         */
        urlParams: {
            regionSlug: TGeoRegionSlug;
            filterSlug?: TFilterSlug;
        };
    };
}

/**
 * Ссылка на страницу поиска, саму ссылку формирует фронт
 */
export interface ISearchHotelsLinkBlock {
    type: ESectionBlock.SEARCH_HOTELS_LINK_BLOCK;
    data: {
        /**
         * Текст ссылки
         */
        text: string;
        /**
         * Параметры для формирования ссылки
         */
        urlParams: ISearchPageIdentifyParams;
    };
}

/**
 * Блок ссылок различных типов
 */
export type TLinkTextBlock =
    | IExternalLinkBlock
    | IHotelLinkBlock
    | IRegionLinkBlock
    | ISearchHotelsLinkBlock
    | IRouteLinkBlock
    | ISearchAviaLinkBlock;

/**
 * Просто текстовый блок, может быть стилизован
 */
export interface IPlainTextBlock {
    type: ESectionBlock.PLAIN_TEXT_BLOCK;
    data: {
        /**
         * Сам текст
         */
        text: string;
        /**
         * Стилизация текста
         */
        styles?: EPlainTextStyle[];
    };
}

/**
 * Блок для отрисовки цены в тексте
 */
export interface IPriceTextBlock {
    type: ESectionBlock.PRICE_TEXT_BLOCK;
    data: {
        /**
         * Цена
         */
        price: IPrice;
    };
}

/**
 * Возможные составляющие текстового блока
 */
export type TTextChildBlock =
    | IPlainTextBlock
    | TLinkTextBlock
    | IPriceTextBlock;

/**
 * Текстовый блок с дочерними элементами различных текстовых типов
 */
export interface ITextBlock {
    type: ESectionBlock.TEXT_BLOCK;
    children: TTextChildBlock[];
}

/**
 * Блок с текстом с заголовком h3(может не быть) и дочерними элементами текстового типа. Объект data может быть пустой
 */
export interface ISubSectionTextBlock {
    type: ESectionBlock.SUB_SECTION_TEXT_BLOCK;
    data: {
        /**
         * Заголовок h3
         */
        title?: string;
    };
    children: ITextBlock[];
}

/**
 * Блок с заголовком и раскрывающимся описанием
 */
export interface ISpoilerBlock {
    type: ESectionBlock.SPOILER_BLOCK;
    data: {
        /**
         * Заголовок(кликабельный)
         */
        title: string;
        /**
         * Подробное описание, рендерится при клике на заголовок только на клиенте!
         */
        description: ITextBlock;
    };
}

/**
 * Группа из двух ссылок, основной и вспомогательной. Ссылки могут быть различный типов.
 */
export interface IGeoLinkGroupBlock {
    type: ESectionBlock.GEO_LINK_GROUP_BLOCK;
    data: {
        /**
         * Основная ссылка(синия)
         */
        mainLink: TLinkTextBlock;
        /**
         * Второстепенная ссылка(серая)
         */
        additionalLink?: TLinkTextBlock;
    };
}

/**
 * Возможные дочерние элементы для SectionTextBlock
 */
export type TSectionTextChildBlock =
    | ISubSectionTextBlock
    | ISpoilerBlock
    | IGeoLinkGroupBlock
    | ITextBlock;

/**
 * Блок с заголовком h2 (может не быть) и дочерними элементами различных типов. Объект data может быть пустой
 */
export interface ISectionTextBlock {
    type: ESectionBlock.SECTION_TEXT_BLOCK;
    data: {
        /**
         * Заголовок h2
         */
        title?: string;

        text?: string;

        /**
         * Заголовок для быстрых ссылок (если не передан - то в навигации не участвует)
         */
        navigationTitle?: string;
    };
    children: TSectionTextChildBlock[];
}
