import {IAffiliateAttribution} from 'common/IAffiliateAttribution';
import {EWhiteLabelBadgeId} from 'common/white-label/common';

import {TDate, ILinguistics, TInteger} from '../../common/common';
import {IHotel} from '../hotel-page-flow/getHotelInfo/models';
import {IHotelOffer} from '../hotel-page-flow/getHotelOffers/models';

/** *****************************************************************************/
/* Общие скалярные типы. */
/** *****************************************************************************/

// Пермалинк. Он же идентификатор отеля.
// Передается строкой, по сути -- ui64-число.
export type TPermalink = string;

// Slug отеля. Он же - текстовый идентификатор отеля.
// Slug может состоять из нескольких частей, разделённых слэшом "/".
// Например, "moscow/holiday-inn"
export type THotelSlug = string;

// Geo Id. Идентификатор региона
export type TGeoId = number;

// Slug региона. Он же - текстовый идентификатор региона.
export type TGeoRegionSlug = string;

/** *****************************************************************************/
/* Общие структурные типы. */
/** *****************************************************************************/

/**
 * Географические координаты. Могут быть сериализованы в строку как "lat,lon".
 * @example {lat: 55.757195, lon: 37.648640}
 *
 * TODO: выбрать один из вариантов ICoordinates из common/common или отсюда
 */
export interface ICoordinates {
    lat: number; // Широта.
    lon: number; // Долгота.
}

/**
 * Параметры ограничивающего прямоугольника для карты.
 */
export interface IBoundingBoxParams {
    /**
     * Строковое представление
     * Если пришло в запросе, то отдаем такой же; иначе вычисляем сами и отдаем фронту.
     *
     * Формат: lon,lat~lon,lat
     * Пример: 37.0408809,55.311850~38.20412732,56.18961995
     */
    bboxAsString: string;

    /**
     * Представление в виде кортежа
     */
    bboxAsStruct: ICoordinates[];
}

// Параметры поиска вариантов размещения.
// _Обязательный_ к указанию вариант.
//
// Пример кодирования в CGI-параметрах:
//   &checkinDate=2019-10-11&checkoutDate=2019-10-13&adults=2&childrenAges[]=1&childrenAges[]=3
export interface IRequiredOfferSearchParams {
    // Дата заселения в гостиницу в формате YYYY-MM-DD.
    checkinDate: TDate;
    // Дата выезда из гостиницы
    checkoutDate: TDate;
    // Количество взрослых.
    adults: number;
    // Массив возрастов детей.
    childrenAges: number[]; // childrenAges[]=1&childrenAges[]=3
}

// Отладочные параметры поиска вариантов размещения.
export interface IDebugOfferSearchParams {
    /** Хост портала, для подмены в офферах. Работает только на тестинге. */
    debugPortalHost?: string;
    /** Если true, то поиск делается в продовом офферкэше. Работает только на тестинге. */
    debugUseProdOffers?: boolean;
}

/**
 * Параметры поиска вариантов размещения.
 *_Опциональный_ к указанию вариант.
 */
export type TOptionalOfferSearchParams = Partial<IRequiredOfferSearchParams>;

/**
 * Параметры идентификации одного отеля
 * Должно быть задано ровно одно поле, иначе - ошибка.
 */
export interface IHotelIdentifier {
    /** Пермалинк отеля. */
    permalink?: TPermalink;
    /** Slug, как альтернатива пермалинку. */
    hotelSlug?: THotelSlug;
}

/**
 * Параметры идентификации одного региона.
 * Не может быть задано несколько идентификаторов
 */
export interface IRegionIdentifier {
    /**
     * Идентификатор региона
     */
    geoId: TGeoId;

    /**
     * Slug региона
     */
    regionSlug: TGeoRegionSlug;
}

// Атрибуция пользователя.
export interface IAttribution extends IAffiliateAttribution {
    utmSource?: string; // Передаётся из URL-параметра utm_source
    utmMedium?: string; // Передаётся из URL-параметра utm_medium
    utmCampaign?: string; // Передаётся из URL-параметра utm_campaign
    utmTerm?: string; // Передаётся из URL-параметра utm_term
    utmContent?: string; // Передаётся из URL-параметра utm_content
    serpReqId?: string; // Передаётся из URL-параметра serpReqId
    gclid?: string; // Передаётся из URL-параметра gclid
    userRegion?: number; // Передаётся из URL-параметра userRegion
    ytpReferer?: string; // Передаётся из URL-параметра ytp_referer
    yclid?: string; // Передаётся из URL-параметра yclid
    fbclid?: string; // Передаётся из URL-параметра fbclid
    metrikaClientId?: string; // Идентификатор метрики, полученный через https://yandex.ru/support/metrica/objects/get-client-id.html
    metrikaUserId?: string; // Идентификатор пользователя UserID по этой инструкции https://yandex.ru/dev/metrika/doc/api2/practice/conversion.html#offline
    referralPartnerRequestId?: string; // Передаётся из URL-параметра referral_partner_request_id
}

// Описание региона.
// TODO(sandello): Требуется указание языка!
// Пример данных: http://geobase.qloud.yandex.ru/v1/linguistics_for_region?id=213&lang=ru
export interface IGeoRegion {
    geoId: TGeoId;
    linguistics: ILinguistics;
    slug?: TGeoRegionSlug;
    type?: TInteger;
}

enum EBreadcrumbType {
    GEO_REGION_BREADCRUMB = 'IGeoRegionBreadcrumb',
    FILTER_BREADCRUMB = 'IFilterBreadcrumb',
    HOTEL_BREADCRUMB = 'IHotelBreadcrumb',
}

interface IGeoRegionBreadcrumb extends IGeoRegion {
    breadcrumbType: EBreadcrumbType.GEO_REGION_BREADCRUMB;
}

interface IFilterBreadcrumb {
    breadcrumbType: EBreadcrumbType.FILTER_BREADCRUMB;
    name: string;
    slug: string;
}

interface IHotelBreadcrumb {
    breadcrumbType: EBreadcrumbType.HOTEL_BREADCRUMB;
    name: string;
    slug: string;
}

type TBreadcrumb = IGeoRegionBreadcrumb | IFilterBreadcrumb | IHotelBreadcrumb;

export interface IBreadcrumbs {
    /** Массив хлебных крошек (содержит только IGeoRegion)
     * @deprecated убрать после перехода на breadcrumbs.items
     * */
    geoRegions: IGeoRegion[];
    /** Массив хлебных крошек
     * для SEO-страниц с фильтром также содержит IFilterBreadcrumbs
     * */
    items: TBreadcrumb[];
}

export interface IOfferSearchProgress {
    /** Закончен ли поиск офферов в связаной группе отелей. */
    finished: boolean;
    /** Общее количество опрашиваевых парнеров */
    partnersTotal: number;
    /** Количество опрошенных парнеров */
    partnersComplete: number;
}

export interface IOfferSearchProgressInfo {
    offerSearchProgress: IOfferSearchProgress;
    nextPollingRequestDelayMs?: number;
}

export enum EGeoLocationStatus {
    UNKNOWN = 'unknown', // Ещё не запрашивали разрешение
    AVAILABLE = 'available', // Геолокация доступна, имеем координаты пользователя
    FORBIDDEN = 'forbidden', // Пользователь явно запретил доступ к геолокации
}

/** *****************************************************************************/
/* Параметры визита и пользователя, см HOTELS-5229
/** *****************************************************************************/

export interface IExtraVisitAndUserParams {
    /** Параметры визита. Могут отсутствовать, если нет ни одного параметра. Передаются в метрику без изменений. */
    visitParams?: Record<string, any>;
    /** Параметры пользователя. Могут отсутствовать, если нет ни одного параметра. Передаются в метрику без изменений. */
    userParams?: Record<string, any>;
}

/** Типы сортировки отзывов */
export enum ETextReviewRankingType {
    TIME = 'byTime', // По времени
    RELEVANCE_ORG = 'byRelevanceOrg', // По релевантности ("интересные" отзывы в топе)
    LIKES_COUNT_DESC = 'byLikesCountDesc', // По убыванию количества положительных лайков
    RATING_ASC = 'byRatingAsc', // По возрастанию рейтинга (от плохого к хорошему)
    RATING_DESC = 'byRatingDesc', // По убыванию рейтинга (от хорошего к плохому)
}

// Комбинированная информация об отеле и о предложениях в нем.
// Используется в блоке похожих отелей, на странице поиска в регионе.
export interface IHotelWithOffers {
    // Информация об отеле.
    hotel: IHotel;
    // Закончился ли поиск предложений конкретно в этом отеле.
    // Если true, то есть поле offers.
    // Если false, то поля offers еще нет.
    searchIsFinished: boolean;
    // Офферы для отображения.
    // Режимы использования:
    // - в похожих отелях: должен быть ровно 0 или 1 оффер.
    // - в странице региона: произвольное, но не очень большое число офферов; первый из них -- главный.
    offers?: IHotelOffer[];
    // Бэджики для отеля
    badges?: IBadge[];
    // Отель нужно выделить особым образом (подсветить/обвести в рамочку).
    // Нужно для редиректов с рекламы конкретных отелей на страницу поиска: TRAVELBACK-1942
    searchedByUser: boolean;
}

/* Тема бейджа, которая приходит с бэка. Переопределяет тему, которая используется в логике фронта. TRAVELFRONT-8587 */
export enum EBadgeTheme {
    GREEN = 'GREEN',
    BLUE = 'BLUE',
    RED = 'RED',
    NEUTRAL = 'NEUTRAL',
    YELLOW = 'YELLOW',
    BLACK = 'BLACK',
    WHITE = 'WHITE',
}

export interface IBadge {
    /** id бэйджа, для того чтобы рисовать разными стилями. */
    id:
        | 'min_price'
        | 'hotel_direct'
        | 'taxi_promocode'
        | 'mir_cashback'
        | 'welcome_promocode'
        | 'ads'
        | 'yandex_eda'
        | 'yandex_plus'
        /** Для отображения партнерских баллов по программе White Label */
        | EWhiteLabelBadgeId;
    /** Текст бэджа */
    text: string;
    /** Данные с описанией акций, которые отображаются в попапах или рядом с бэйджом */
    additionalPromoInfo?: {
        title?: string;
        text: string;
        link?: {
            text: string;
            url: string;
        };
    };
    theme?: EBadgeTheme;
}
