import {IHotelImage} from '../getHotelImages/models';
import {ISeoInfo} from '../../../common/seo-pages/ISeoInfo';
import {TPermalink, THotelSlug, ICoordinates} from '../../common/common';

export enum ELayoutType {
    META = 'M',
    BOOK = 'B',
}

export enum ETransportStationType {
    METRO = 'METRO',
    OTHER = 'OTHER',
}

export enum EMainAmenityId {
    WI_FI = 'wi_fi',
    POOL = 'pool',
    CAR_PARK = 'car_park',
    AIR_CONDITIONING = 'air_conditioning',
    GYM = 'gym',
    PAYMENT_BY_CARD = 'payment_by_credit_card',
    BEACH_LINE = 'beach_line',
}

export interface IMetroLine {
    id: string; // id линии "100000267"
    name: string; // "Первая линия"
    color?: string; // цвет линии, если доступен
}

export interface ITransportStation {
    id: string; // id станции "station__9765108"
    type: ETransportStationType;
    name: string; // "Площадь 1905г"
    coordinates: ICoordinates; // Расположение
    distanceMeters: number; // Расстояние в метрах, "1114.87"
    distanceText: string; // Человекочитаемое расстояние, "1,1 км"
    metroLine?: IMetroLine; // Если метро - то информация о линии
}

export interface IHotel {
    /** Пермалинк */
    permalink: TPermalink;
    /** Главный slug отеля. Может отсутствовать. */
    hotelSlug?: THotelSlug;
    /**
     * Наименование отеля.
     * @example "Артист"
     */
    name: string;
    /** Категория отеля (аналог рубрики в Справочнике). */
    category: IHotelCategory;
    /** Координаты отеля */
    coordinates: ICoordinates;
    /**
     * Адрес отеля.
     * @example "Ул. Добролюбова, д. 11"
     */
    address: string;
    /**
     * Количество звезд. От 1 до 5. Целое число.
     * Может отстустовать, не у всех отелей есть звёзды.
     * @example 2
     */
    stars?: number;
    /**
     * Рейтинг. От 1 до 5. Дробное число, 1 знак после запятой.
     * Может отстустовать, не у всех отелей есть рейтинг.
     * @example 3.4
     */
    rating?: number;

    /**
     * Заботливая галка. "Информация об организации подтверждена владельцем"
     * Отсутствует для похожих отелей.
     */
    hasVerifiedOwner?: boolean;

    /** Информация о ближайших остановках транспорта
     * Отсутствует для похожих отелей.
     */
    nearestStations?: ITransportStation[];

    /** Общее число отзывов. */
    totalTextReviewCount: number;

    /** Общее количество фотографий.
     * Отсутствует для похожих отелей.
     */
    totalImageCount?: number;
    /** Фотографии.
     * Количество переданных фотографий зависит от контекста использования IHotel.
     * К примеру, для страницы отеля сразу будет указано несколько фотографий.
     * Для карты, для списка -- одна фотография. */
    images: IHotelImage[];

    /** Основные удобства. Отображаются с иконкой и без группировки. */
    mainAmenities?: IMainHotelAmenity[];
    /** Группы удобств */
    amenityGroups?: IHotelAmenitiesGroup[];

    /** Название города
     * @deprecated убрать после перехода на hotelLocation
     * Отображается на SEO странице региона вместо информации о ближайших остановках транспорта
     * Отсутствует для загородных отелей
     * */
    cityName?: string;

    /** Населённый пункт, в котором находится отель
     * Отображается на SEO странице региона вместо информации о ближайших остановках транспорта
     * @example "г. Сергиев Посад"
     * */
    location?: string;

    /**
     * Приблизительное расстояние от пользователя до отеля (в метрах). Передаётся только при сортировке по расстоянию.
     */
    distanceMeters?: number;

    /**
     * Приблизительное расстояние от пользователя до отеля (в виде текста с единицей измерения). Передаётся только при сортировке по расстоянию.
     */
    distanceText?: string;

    /**
     * Содержится ли отель в избранном? Null означает, что информации нет и сердечко рисовать не нужно.
     */
    isFavorite?: boolean;

    /**
     * Геофича отеля
     */
    geoFeature?: IHotelGeoFeature | null;
}

/**
 * Данные, которые нужны на странице отеля и не нужны на пинах и странице поиска.
 * XXX(sandello): В этом году описание будет пустовать. Нет данных.
 */
export interface IHotelDescription {
    /** Описание отеля, полное.
     * Пример: "Гранд-отель самый лучший в мире. Круассаны, чурчхела, пахлава. Заходи, плати, отдыхай, спи, кушай." */
    text?: string;
}

export interface IHotelAmenitiesGroup {
    /** Id. Машино-читаемая строка */
    id: string;
    /**
     * Человеко-читаемое название. Для отрисовки.
     * @example "Услуги и удобства"
     */
    name: string;
    /** Удобства группы */
    amenities: IHotelAmenity[];
}

export interface IMainHotelAmenity {
    /** Id. Машино-читаемая строка. На её основе рисуется иконка.  */
    id: EMainAmenityId;
    /** Человеко-читаемое название. Для отрисовки.
     * Пример: "химчиста", "душ", "парковка". */
    name: string;
}

export interface IHotelAmenity {
    /** Id. Машино-читаемая строка. На её основе рисуется иконка.
     * TODO(alexcrush): Составить список well-known values. */
    id: string;
    /** Человеко-читаемое название. Для отрисовки.
     * Пример: "химчистка", "душ", "парковка". */
    name: string;
}

export interface IRoomAmenity {
    /**
     * Уникальный идентификатор
     */
    id: string;

    /**
     * Машино-читаемая строка. На её основе рисуется иконка.
     */
    icon: string;

    /**
     * Человеко-читаемое название. Для отрисовки.
     * Пример: "35 кв.м.", "Холодильник", "Мини-бар: за доплату".
     */
    name: string;
}

export interface IHotelCategory {
    /** Id. Машино-читаемая строка. На её основе рисуется иконка.
     * Для "похожих" отелей - отсутствует.
     * TODO(alexcrush): Составить список well-known values. */
    id?: string;
    /** Человеко-читаемое название. Для отрисовки.
     * Пример: "гостиница", "хостел", "апарт-отель". */
    name: string;
}

export interface IFeatureRatingInfo {
    /** Id. Машино-читаемая строка */
    id: string;
    /** Человеко-читаемое название. Для отрисовки. */
    name: string;
    /** Процент позитивных оценок - число от 0 до 100*/
    positivePercent: number;
}

export interface IRatingsInfo {
    /** Отзыв-тизер. На макете -- сверху справа, "80% гостям что-то понравилось."
     * На бекенде -- информацию брать из сниппета feature_ugc_answers. */
    teaser?: string;

    /** Список оценок отеля из правой колонки макета. */
    featureRatings: IFeatureRatingInfo[];
}

export interface IHotelSeoInfo extends ISeoInfo<IHotelSchemaOrgInfo> {}

/* Schema.org-разметка страницы */
export interface IHotelSchemaOrgInfo {
    // URL страницы и URL карты формируется фронтом, чтобы не нагружать бэк знанием формата урла портала/карты

    /** Имя отеля
     * https://schema.org/name
     */
    name: string;

    /** абсолютный урл картинки отеля
     * https://schema.org/image
     */
    image: string;

    /** ценовой диапозон в формате 500р. - 1000р.
     *  https://schema.org/priceRange
     * Может отсутствовать, если неизвестен
     */
    priceRange?: string;

    /** адрес отеля
     * https://schema.org/address
     */
    address: string;

    /** Рейтинг. От 1 до 5. Дробное число, 1 знак после запятой.
     * https://schema.org/ratingValue
     */
    ratingValue: number;

    /** Число отзывов.
     * https://schema.org/reviewCount
     */
    reviewCount: number;
}

export interface IHotelGeoFeature {
    /**
     * Уникальный id геофичи
     */
    id: string;

    /**
     * Идентификатор иконки для отрисовки
     */
    icon: string;

    /**
     * Строковое представление геофичи (например, "1км от центра" или "Рядом аэропорт")
     */
    name: string;
}
