import {IPrice, TInteger, TInstantDateTime} from '../../../common/common';
import {IRoomAmenity} from '../getHotelInfo/models';
import {IOfferSearchProgress, IBadge} from '../../common/common';
import {IHotelImage} from '../getHotelImages/models';

export type TOperatorId = string;

export type TRoomId = string;

export type TOfferMeal =
    | 'AI'
    | 'UAI'
    | 'LAI'
    | 'BB'
    | 'FB'
    | 'HB'
    | 'RO'
    | 'BD'
    | 'UNKNOWN'; // Id типа питания

export type TOperatorById = Record<TOperatorId, IHotelsOperator>;

export enum EOfferGroupType {
    ROOMS = 'ROOMS',
}

export enum ERefundType {
    FULLY_REFUNDABLE = 'FULLY_REFUNDABLE',
    REFUNDABLE_WITH_PENALTY = 'REFUNDABLE_WITH_PENALTY',
    NON_REFUNDABLE = 'NON_REFUNDABLE',
}

export enum EBannerType {
    NONE = 'NONE', // Баннер не нужен
    HOTEL_DIRECT = 'HOTEL_DIRECT', // Напрямую от отеля
    MIR_CASHBACK = 'MIR_CASHBACK', // Кешбек на карту Мир по акции (HOTELS-5171)
    POSSIBLE_MIR_CASHBACK = 'POSSIBLE_MIR_CASHBACK', // Если поменять даты, то может быть кешбек на карту Мир по акции (HOTELS-5171)
    WELCOME_PROMOCODE = 'WELCOME_PROMOCODE', // Промокод на первый заказ, TRAVELMARKETING-43
}

export enum EPansionAggregate {
    NO_PANSION_AVAILABLE = 'NO_PANSION_AVAILABLE', // Нет вариантов с питанием
    PANSION_AVAILABLE = 'PANSION_AVAILABLE', // Есть варианты с питанием
}

export enum ECancellationInfoAggregate {
    FULLY_REFUNDABLE_AVAILABLE = 'FULLY_REFUNDABLE_AVAILABLE', // Есть варианты с бесплатной отменой
    REFUNDABLE_WITH_PENALTY_AVAILABLE = 'REFUNDABLE_WITH_PENALTY_AVAILABLE', // Есть варианты с отменой со штрафом
    NON_REFUNDABLE_AVAILABLE = 'NON_REFUNDABLE_AVAILABLE', // Доступны только неотменяемые предложения, отмена не предусмотрена
}

export enum EDiscountReason {
    YANDEX_HOTEL = 'YANDEX_HOTEL', // Скидка в Я.Отель, a.k.a. Будапешт
}

export interface IHotelPartnerOffersInfo {
    /** Идентификатор оператора */
    operatorId: TOperatorId;

    /** Агрегированная информация о наличии питания в предложениях этого партнёра. Если отсутствует - не рисуем соответствующую надпись. */
    pansionAggregate?: EPansionAggregate;

    /** Агрегированная информация о наличии отменяемых предложений у этого партнёра. Если отсутствует - не рисуем соответствующую надпись. */
    cancellationInfoAggregate?: ECancellationInfoAggregate;

    /** Этот оффер нужно показать в случае отображения только одной партнёрской цены */
    defaultOffer?: IHotelOffer;

    /**
     * инфомация о типе питания из офера
     */
    defaultOfferPansion?: IHotelOfferMeal | null;

    /**
     * Информация об отмене из офера
     */
    defaultOfferCancellationInfo?: ICancellationInfo | null;
}

export interface IAggregatedOfferInfo {
    /** Минимальная цена по всем офферам */
    minPrice: IPrice;

    /** Максимальная цена по всем офферам */
    maxPrice: IPrice;

    /** Агрегированная информация о наличии питания в офферах. Если отсутствует - не рисуем соответствующую надпись. */
    pansionAggregate?: EPansionAggregate;

    /** Агрегированная информация о наличии отменяемых предложений в офферах. Если отсутствует - не рисуем соответствующую надпись. */
    cancellationInfoAggregate?: ECancellationInfoAggregate;
}

export interface IHotelOffersInfo {
    /**
     * Прогресс поиска предложений.
     * */
    offerSearchProgress: IOfferSearchProgress;
    nextPollingRequestDelayMs?: number; // Заполнено, если offerSearchProgress.finished == false. Минимальное значение - 200ms

    /*
     * Если offerSearchProgress.finished == false, то все поля ниже отсутствуют.
     * Если offerSearchProgress.finished == true, то все поля ниже присутствуют.
     */

    /** Баннер-шапка над отелем (например синяя "Напрямую от отеля") */
    bannerType?: EBannerType;

    /**
     * Агрегированная информация обо всех офферах. Нужна для кнопки "посмотреть все оффера" (на ней либо пишется "от X рублей", либо рядом с ней пишется информация о диапазоне цен и питании).
     */
    aggregatedOfferInfo?: IAggregatedOfferInfo;

    /**
     * Предложения, показывающиеся в основном списке, над блоком предложений партнёров.
     *
     * Если активен и успешен матчинг партнёрских офферов в команты - в это поле попадают все оффера.
     * Если матчинг не активен, то тут только BoY предложения.
     * Только оффера из этого списка участвуют в группировке по комнатам (если есть поле rooms).
     *
     * Может отсутствовать, тогда:
     * 1. В случае rooms==null не рисуем блок яндексовых цен (комнаты или плоский список).
     * 2. В случае rooms!=null во всех комнатах просто не будет предложений, но сами комнаты рисуем.
     */
    mainOffers?: IHotelOffer[];

    /**
     * Предложения не-BoY партнёров. Рисуются в конце списка, не участвуют в группировке по комнатам.
     * Может отсутствовать, тогда не рисуем блок партнёрских цен.
     * Если активен и успешен матчинг партнёрских предложений, то это поле пусто.
     */
    partnerOffers?: IHotelPartnerOffersInfo[];

    /**
     * Список комнат. Порядок комнат на странице задаётся порядком в этом списке.
     * Офферы внутри комнат должны быть взяты из списка mainOffers и иметь тот же относительный порядок, что в списке mainOffers.
     *
     * Может отсутствовать, тогда рисуем вариант страницы без комнат.
     */
    rooms?: IHotelRoom[];

    /** Словарь Информации об операторах. */
    operatorById?: TOperatorById;

    /**
     * Общее число предложений о размещении в данном отеле.
     * Используется для подписей вида "Еще N предложений от M партнеров."
     */
    offerCount?: number;
    /**
     * Общее число партнеров, предоставляющих информацию о размещении в данном отеле.
     * Используется для подписей вида "Еще N предложений от M партнеров."
     * Для мета-варианта и для BoY-варианта всегда показывает общее число партнеров по количеству провязок отеля.
     * Может не совпадать с количеством опрашиваемых при поиске партнеров.
     */
    operatorCount?: number;
}

/** Предложение отеля */
export interface IHotelOffer {
    /** Идентификатор предложения */
    id: string;
    /**
     * Наименование предложения
     * @example "Стандартный номер с 1 кроватью или 2 креслами"
     */
    name: string;
    /** Стоимость по данному офферу */
    price: IPrice; // 3000 Р
    /** Идентификатор комнаты */
    roomId?: TRoomId;

    /** Идентификатор оператора */
    operatorId: TOperatorId;
    /**
     * Ссылка на бронирование, куда переходить при клике на данный оффер.
     * @example "https://booking.com/..."
     */
    landingUrl: string;
    /** Политика отмены */
    cancellationInfo?: ICancellationInfo;
    /** Тип питания */
    mealType?: IHotelOfferMeal;
    /** Флаг предложения от Яндекса */
    yandexOffer?: boolean;
    /** Token для корректного формирования ссылки на BoY страницу отеля. Есть только если yandexOffer == true */
    token?: string;
    // Бэджики для данного оффера
    badges?: IBadge[];

    /* Информация о скидке. Может отсутствовать */
    discountInfo?: IDiscountInfo;
    /* Баллы плюса, которые возможно будут начислены. При отсутствии поля никакие баллы начислены не будут */
    offerYandexPlusInfo?: IOfferYandexPlusInfo;
}

export interface IOfferYandexPlusInfo {
    /* количество баллов для начисления */
    points: TInteger;
    /* True, если оффер участвет в акции: доступно начислены или списание; False если нет */
    eligible: boolean;
}

export interface IDiscountInfo {
    /* Базовая цена (до скидки) */
    strikethroughPrice: IPrice;
    /* Процент скидки по шкале от 0 до 100. Может отсутствовать, если скидка не выражается в процентах */
    percent?: string | number;
    /* Причина возникновения скидки. Может отсутствовать, если причина скидки неизвестна */
    reason?: EDiscountReason;
}

export interface IHotelOfferMeal {
    /** Идентификатор типа питания. Используется для отрисовки иконки */
    id: TOfferMeal;
    /** Наименование типа питания
     * @example Завтрак
     * @example Ужин
     */
    name: string;
}

export interface IRefundRule {
    startsAt: TInstantDateTime;
    endsAt: TInstantDateTime;
    type: ERefundType;
    penalty?: IPrice;
}

export interface ICancellationInfo {
    /** Бесплатная отмена (есть/нет). */
    hasFreeCancellation: boolean;
    // Отменяемость на текущий момент
    refundType: ERefundType;
    /**
     * Подробные политики отмены.
     * - Если есть, то рисуется иконка с вопросиком, и в тултипе выводится подробная информация о штрафах.
     * - Есть нет, то ничего не рисуется
     */
    refundRules?: IRefundRule[];
}

export interface IHotelsOperator {
    /**
     * Id оператора.
     * @example "BookingId"
     * @example "AgodaId"
     * @example "YandexId"
     */
    id: string;
    /**
     * Наименование оператора.
     * @example "Букинг"
     * @example "Яндекс Путешествия"
     */
    name: string;
    /**
     * Грин-урл.
     * TODO(sandello): Посмотреть, используется ли на макетах.
     */
    greenUrl: string;
    /** Иконка для отрисовки. */
    iconUrl: string;
}

export interface IRoomAmenityGroup {
    /** Id. Машино-читаемая строка */
    id: string;
    /**
     * Машино-читаемая строка. На её основе рисуется иконка.
     */
    icon: string;
    /**
     * Человеко-читаемое название. Для отрисовки заголовка группы удобств.
     * @example "Ванная комната"
     */
    name: string;
    /** Удобства группы */
    amenities: IRoomAmenity[];
}

/**
 * Конкретный тип кровати и количество кроватей такого типа
 */
export interface IRoomBedConfigurationItem {
    /** Id. Машино-читаемая строка */
    id: string;

    /** Идентификатор иконки */
    icon: string;

    /** Название типа кровати в начальной форме
     *  Пример:
     *  "Односпальная кровать"
     */
    nameInitialForm: string;

    /** Название типа кровати в согласованной с quantity форме
     *  Примеры:
     *  quantity = 1; nameInflectedForm = "Односпальная кровать"
     *  quantity = 4; nameInflectedForm = "Односпальные кровати"
     *  quantity = 10; nameInflectedForm = "Односпальных кроватей"
     */
    nameInflectedForm: string;

    /** Количество кроватей такого типа в номере */
    quantity: number;
}

/**
 * Группа кроватей в номере
 */
export interface IRoomBedGroup {
    /** Id. Машино-читаемая строка */
    id: string;

    /**
     * Структура группы
     * Элементы объединяются через "И", т.е. в комнате присутствуют сразу все перечисленные кровати.
     */
    configuration: IRoomBedConfigurationItem[];
}

/**
 * Единица измерения площади номера
 */
export enum ERoomAreaUnits {
    SQUARE_METERS = 'SQUARE_METERS',
}

/**
 * Площадь номера
 */
export interface IRoomArea {
    /** Значение площади */
    value: number;

    /** Единица измерения */
    unit: ERoomAreaUnits;
}

export interface IHotelRoom {
    /** Id комнаты. */
    id: TRoomId;
    /**
     * Человеко-читаемое название. Для отрисовки.
     * @example "Стандартный номер"
     */
    name: string;
    /** Фотографии номера */
    images?: IHotelImage[];
    /** Описание номера */
    description?: string;

    /**
     * Группы кроватей. Объединяются через "ИЛИ", т.е. единовременно в номере доступна только одна из групп.
     */
    bedGroups?: IRoomBedGroup[];

    /**
     * Площадь номера
     */
    area?: IRoomArea;

    /**
     * Удобства в номере
     */
    mainAmenities?: IRoomAmenity[];
    /**
     * Группы удобств в номере
     */
    amenityGroups?: IRoomAmenityGroup[];

    /** Агрегированная информация о наличии питания в предложениях этого номера. Если отсутствует - не рисуем соответствующую надпись. */
    pansionAggregate?: EPansionAggregate;

    /** Агрегированная информация о наличии отменяемых предложений в этом номере. Если отсутствует - не рисуем соответствующую надпись. */
    cancellationInfoAggregate?: ECancellationInfoAggregate;

    /**
     * Бэйджи комнаты
     */
    badges?: IBadge[];
}
