import {ECurrencyType} from '../../../common/common';

/**
 * Баннер на странице поиска
 */
export enum ESearchBannerType {
    MIR = 'MIR',
    MIR_ENABLED = 'MIR_ENABLED',
}

export enum EStartSearchReasonType {
    /** Самый первый поиск при маунте страницы. Заход в новой вкладке, перезагрузка страницы или браузерный назад-вперед(в рамках других страниц). */
    MOUNT = 'mount',
    /** Изменились query параметры. Форма поиска или браузерный назад-вперед(в рамках страницы поиска). */
    QUERY_BY_LOCATION = 'queryByLocation',
    /** Изменился bounds карты. */
    MAP_BOUNDS = 'mapBounds',
    /** Нажатие кнопок назад-вперед под списком. */
    NAVIGATION_TOKEN = 'navigationToken',
    /** Применение фильтров. */
    FILTERS = 'filters',
    /** Использование поискового текстового фильтра. */
    FILTERS_TYPE_SEARCH_TEXT = 'filtersTypeSearchText',
    /** Изменение сортировки. */
    SORT = 'sort',
}

export enum EResetFilterActionEffect {
    ALL = 'reset:all',
    SIMPLE = 'reset:simple',
}

export enum EResetFilterReason {
    ZOOMING_MAP = 'zooming_map',
    FILTERS = 'filters',
}

/** Атомарный фильтр; типа "star:three". */
export type TFilterAtom = string;

/** Состояние всех фильтров на странице поиска в регионе. */
export interface IFilterParams {
    filterAtoms?: TFilterAtom[]; // "star:three,star:four,rating:4+,wi_fi:1"
    filterPriceFrom?: number;
    filterPriceTo?: number;
    // Фильтрация по текущему geo id поиска. True - фильтрует, False - нет. Если флаг не указан, то бэк выставляет дефолтное значение true
    onlyCurrentGeoId?: boolean;
}
export type TSortId = string;

export interface ISortParams {
    // Id выбранного типа сортировки
    selectedSortId?: TSortId;

    // Центр сортировки (например для сортировки по расстоянию). Нужен не для всех сортировок.
    // lon,lat (например 37.0408809,55.311850)
    sortOrigin?: string;
}

// Базовый фильтр.
export interface IBasicFilter {
    // Уникальный идентификатор, нужен для оптимизации верстки
    id: string;
    // Название.
    name: string;
    // Эффект, как отображать этот фильтр.
    // Примеры: подкраска рейтинга, спец-значок для "только спецпредложения".
    // Примеры: для быстрого фильтра -- какой-нибудь спец-цвет или другая штука.
    effect?: string;

    // Если enabled == true, то можно на контрол нажимать.
    // Если enabled == false, то нажимать нельзя (будет пустая выдача).
    enabled: boolean;

    // hint -- число отелей, которое будет у пользователя при выборе фильтра.
    // NB: Это информативное для фронта поле.
    // Если фильтр будет расширять ответ, то строка отформатирована как "+100".
    // Если фильтр будет сужать ответ, то строка отформатирована как "50".
    hint: string;

    // Семантику см. в README.md.
    atoms: TFilterAtom[];
}

// Группа базовых фильтров.
export interface IBasicFilterGroup {
    // Уникальный идентификатор, нужен для оптимизации верстки
    id: string;
    // Название.
    name: string;
    // SINGLE -- Radio-селект; можно выбрать только одну позицию.
    // MULTI -- Multi-Checkbox-селект; можно выбрать много позиций.
    type: 'SINGLE' | 'MULTI';

    // Что стоит проверять на бекенде:
    // \forall i1, i2 \in Items: i1.Atoms \cap i2.Atoms == \emptyset
    // (любые две позиции обладают непересекающимся набором атомов)
    items: IBasicFilter[];
}

// Быстрый фильтр
export type TQuickFilter = IQuickControl & {
    // Название.
    name: string;
    // всплывающая подсказка. Если отсутствует - то её нет
    hint?: string;

    // Для определения факта нажатости быстрофильтра:
    // если в текущем стейте фильтра все атомы из atomsOn есть, и нет ни одного из atomsOff - то быстрофильтр нажат
    // При нажатии на быстрофильтр: все атомы из atomsOff - удаляем из стейта, Все атомы из atomsOn - добавляем в стейт.
    // При отжатии быстрофильтра: Все атомы из atomsOn удаляем из стейта. atomsOff в стейте быть не могло, иначе фильтр бы не был нажат
    atomsOn: TFilterAtom[];
    atomsOff: TFilterAtom[];
};

export interface IPriceFilter {
    // Величина, которую стоит показать пользователю, если он НЕ взаимодействовал с левой границей фильтра
    // (то есть filterPriceFrom не указан).
    // Сейчас рисуется как "3 000 Р", но есть вопросы к дизайну.
    // Также можно использовать для оценки диапазона фильтра по цене.
    minPriceEstimate: number;
    // Величина, которую стоит показать пользователю, если он НЕ взаимодействовал с правой границей фильтра
    // (то есть filterPriceTo не указан).
    // Сейчас рисуется как "40 000+ Р".
    // Также можно использовать для оценки диапазона фильтра по цене.
    maxPriceEstimate: number;

    // Валюта (для конструирования IPrice-ов).
    currency: ECurrencyType;

    // [x_0, x_1, x_2, ..., x_{N-1}], то есть N значений.
    //
    // Предположительно, x_0 == 0 всегда.
    //
    // Bounds описывает набор границ ценовых корзин, где расчитана величина гистограммы.
    // Если гистограмму рисовать не надо, то Bounds отсутствует.
    histogramBounds?: number[];
    // [c_0, c_1, ..., c_{N-1}], то есть N значения.
    //
    // c_0 -- количество отелей с ценой от x_0 до x_1
    // c_i -- количество отелей с ценой от x_i до x_{i+1}
    // c_{N-1} -- количество отелей с ценой от x_{N-1} до +infty
    //
    // Counts может отсутствовать -- тогда столбики рисовать не надо!
    histogramCounts?: number[];
    //
    //          |     |     |     |     |     |     |     |     |     |         |         |
    //      c_0 |     | c_1 |     | c_2 |     | c_3 |     | c_4 |     | c_{N-2} |         | c_{N-1}
    // x_0 ------ x_1 ------- x_2 ------- x_3 ------- x_4 ------- ... ----------- x_{N-1} ---------- +infty
    //
}

export interface IDetailedFilterItem {
    // Что должно быть в этом элементе
    // PRICE - фильтр по цене, надо взять его из IFilterInfo
    // GROUP - обычная группа фильтров (передаётся в detailedFilters)
    type: 'PRICE' | 'GROUP';

    // Заполняется, если type == 'GROUP'
    detailedFilters?: IBasicFilterGroup;
}

export interface IDetailedFiltersBatch {
    items: IDetailedFilterItem[];
}

export interface IResetFilterAction {
    // Название действия
    name: string;
    // Тип визуального отображения
    effect: EResetFilterActionEffect;
    // Список id атомов фильтрации, которые нужно выключить при выборе текущего действия
    atomsOff: string[];
    // Нужно ли сбрасывать фильтр по цене. Если true - сбрасывать выставленные значения фильтра, false - не изменять значения фильтра
    needResetFilterPrice: boolean;
    // Нужно ли сбрасывать фильтр по geo id. Если true - выставить значение фильтра false, если false - не изменять значения фильтра
    needResetGeoIdFilter: boolean;
}

export interface IResetFilterInfo {
    // Список возможных действий для сброса фильтров
    actions: IResetFilterAction[];
    // Причина, по которой нужен сброс фильтрации
    resetFilterReason: EResetFilterReason;
}

export interface IGeoIdFilter {
    // Включен ли фильтр или нет
    selected: boolean;
    // Название фильтра
    name: string;
}

export interface IFilterInfo {
    params: IFilterParams;

    // Быстрые фильтры. Не меняются по ходу поллинга.
    quickFilters: TQuickFilter[];
    // Список фильтров
    // Как только появились в ответе, дальше структура уже не будут меняться в рамках этого поллинга (однако hint и enabled менятся могут).
    // Обязательно присутствуют в завершённом поллинге.
    detailedFilters?: IDetailedFilterItem[];
    // Фильтры под выпадушкой - новые, разбитые на группы для отрисовки по колонкам на десктопе
    // Как только появились в ответе, дальше структура уже не будут меняться в рамках этого поллинга (однако hint и enabled менятся могут).
    // Обязательно присутствуют в завершённом поллинге.
    detailedFiltersBatches?: IDetailedFiltersBatch[];

    // Ценовой фильтр. Не меняется по ходу поллинга.
    priceFilter?: IPriceFilter;

    // Информация для сброса фильтров. Если null, то блок сброса фильтров не рисуем
    resetFilterInfo: IResetFilterInfo | null;

    geoIdFilter: IGeoIdFilter | null;
}

export interface ISearchControlInfo {
    quickControls: (TQuickFilter | TQuickPriceFilter | TQuickSort)[];
}

export enum EQuickControlType {
    QUICK_FILTER = 'quick-filter',
    QUICK_FILTER_PRICE = 'quick-price-filter',
    QUICK_SORT = 'quick-sort',
}

export interface IQuickControl {
    // Уникальный идентификатор, нужен для оптимизации верстки
    id: string;
    type: EQuickControlType;
    // Эффект, как отображать этот фильтр.
    // Примеры: подкраска рейтинга, спец-значок для "только спецпредложения".
    // Примеры: для быстрого фильтра -- какой-нибудь спец-цвет или другая штука.
    effect?: string;
    // Если enabled == true, то можно на контрол нажимать.
    // Если enabled == false, то нажимать нельзя (будет пустая выдача).
    enabled: boolean;
}

export type TQuickPriceFilter = IQuickControl & {
    // Название
    name: string;
};

export type TQuickSort = IQuickControl;

export interface ISortType {
    // Идентификатор типа сортировки
    id: string;

    // Название типа сортировки, которое увидит пользователь, если интерфейс НЕ поддерживает сгруппированное отображение сортировок (мобильная версия)
    name: string;

    // Информация о том, какого рода этот тип сортировки: по возрастанию, по убыванию и т.д.
    // В логике не участвует и используется только для отображения пользователю вспомогательной информации (например значка рядом с группой сортировки).
    // Пример:
    // "По цене" - NONE
    // "По цене ⬀" - ASCENDING
    // "По цене ⬂" - DESCENDING
    hint: 'NONE' | 'ASCENDING' | 'DESCENDING';
}

export interface ISortTypeGroup {
    // Идентификатор группы сортировки
    id: string;

    // Название группы сортировки, которое увидит пользователь, если интерфейс поддерживает сгруппированное отображение сортировок (десктопная версия)
    name: string;

    // Требует передачи координат пользователя. То есть перед включением этой сортировки придётся запросить у пользователя доступ к геолокации
    requiresGeoLocation: boolean;

    // Типы сортировки, которые относятся к этой группе. Список не пуст.
    // Порядок элементов важен: при выборе данной группы, активируется сортировка sortTypes[0], переключаются типы по порядку циклически.
    sortTypes: ISortType[];
}

export interface ISortInfo {
    // Id выбранного типа сортировки
    selectedSortId?: TSortId;

    // Центр сортировки (например для сортировки по расстоянию). Нужен не для всех сортировок.
    // lon,lat (например 37.0408809,55.311850)
    // Нужно пробросить в url и в следующие запросы (в том числе удалить из url-а, если здесь он null).
    sortOrigin?: string;

    // Доступные группы сортировки. Обязательно содержит группу, в которой есть тип с id==selectedSortId, если selectedSortId != null
    // Список может быть пуст, тогда блок с сортировками отображать не нужно.
    availableSortTypeGroups: ISortTypeGroup[];
}

export interface ITimingInfo {
    // Время обработки текущего запроса бекендом (в миллисекундах)
    currentRequestDurationMs: number;
}
