import {IPrice, TDate, TInstantDateTime} from 'common/common';

import {ETransportType} from './common';

enum ESegmentEventType {
    FACT = 'fact',
    POSSIBLE_DELAY = 'possible_delay',
    CANCELLED = 'cancelled',
}

enum EStationType {
    RAILROAD = 'train',
    PLANE = 'plane',
    BUS = 'bus',
    WATER = 'water',
}

/** Подтип страницы станции ("поезда", "электрички") */
enum EStationSubtype {
    TRAIN = 'train',
    SUBURBAN = 'suburban',
    SCHEDULE = 'schedule',
    PLANE = 'plane',
    TABLO = 'tablo',
}

enum ESegmentSubtypeCode {
    LASTOCHKA = 'last',
    LASTOCHKA_DAL = 'lastdal',
}

enum EOrderUrlOwner {
    TRAINS = 'trains',
    UFS = 'ufs',
}

enum EHttpMethod {
    GET = 'GET',
    POST = 'POST',
    PUT = 'PUT',
    DELETE = 'DELETE',
}

/** Информация об опозданиях для пунктов прибытия/отправления */
interface ISegmentEvent {
    type: ESegmentEventType;
    minutesFrom: number;
    minutesTo: number;
    minutesFromNew: number;
    minutesToNew: number;
}

interface ISegmentCompany {
    hidden: boolean;
    id: number;
    shortTitle: string;
    title: string;
    url: string;
    ufsTitle?: string;
    yandexAviaUrl?: string | null;
}

interface ISegmentStation {
    id: number;
    country: {
        /** @example 'RU' */
        code: string;
        /** @example 225 */
        id: number;
    } | null;
    /** @example 'Крымская' */
    popularTitle: string;
    /** @example 10997 */
    settlementId: number;
    /** @example 'Europe/Moscow' */
    timezone: string;
    /** @example 'Крымская' */
    title: string;

    /** @example 'Europe/Moscow' */
    railwayTimezone?: string;
    platform?: string;
    ufsTitle?: string | null;
    codes?: {
        /** @example '2064320' */
        express?: string;
    };
    /** @example значение id, пришедшее из ручки search, в случае, если партнер автобусов вернул в качестве id данные города или несовпадающей с нашей станции. */
    idFromSearchApi?: number;
    /** Это свойство есть в ответе ручки search, но его может не быть в сегментах тарифов */
    pageType?: EStationType;
    mainSubtype?: EStationSubtype;
}

interface ISegmentTransport {
    code: ETransportType;
    id: number;
    title: string;
    model?: {
        title: string;
    };
    subtype?: {
        // Информация о подтипе транспорта. Напрмиер здесь, содержится информация о ласточках
        title?: string; // 'Ласточка'
        titleColor?: string; // '#FF7F44'
        id?: number; // 22
        code?: ESegmentSubtypeCode;
    };
}

interface ISegmentFacility {
    code: string;
    icon: string;
    title: string;
}

interface ICancelledSegment {
    toTitleGenitive: string; // "Подольска",
    fromTitleGenitive: string; // "Нахабина"
}

interface ISegmentThreadBase {
    uid: string;
    title: string;
    beginTime: string | null; // Время начала движения для интервальных рейсов. Например "07:00:00"
    endTime: string | null; // Время завершения движения для интервальных рейсов. Например "22:00:00"
    comment: string; // Комментарий перевозчика
    density: string; // Есть в интервальных рейсах. Пример: "маршрутное такси раз в 10-20 минут"
    displaceYabus: null | true; // Признак того что к данному сегменту можно клеить цены от Яндекс Автобусов
    isExpress: boolean;
    isAeroExpress: boolean;
    isBasic: boolean; // Не используется в текущем коде
    number: string;
    schedulePlanCode: null | string;
    deluxeTrain?: {
        id: number;
        title: string;
        shortTitle: string;
        pagePath: string;
        isDeluxe: boolean;
        isHightSpeed: boolean;
    };
    canonicalUid?: string;
    /* Названия станций, на участке между которыми есть отмена. Приходит только если присутствует свойство cancelled */
    cancelledSegments?: ICancelledSegment[];
    /* Признак, что на данном сегменте есть отмена. true, если отменён полностью; - false, если отменён не полностью; отсутствует, если отмен нет. */
    cancelled?: boolean;
}

interface ISegmentThread extends ISegmentThreadBase {
    firstCountryCode: string;
    lastCountryCode: string;
}

interface ISegmentTariffSuplier {
    code: string;
    logo: string;
}

interface ISuburbanCategoryItem {
    description: string;
    title: string;
    url: string;
    isMain: boolean;
    order: number;
    price?: IPrice;
}

interface ISegmentTariffs<TariffClasses> {
    classes: Record<string, TariffClasses>;
    suburbanCategories?: Record<string, ISuburbanCategoryItem[]>;
    electronicTicket?: boolean | null;
    key?: string;
    suplier?: ISegmentTariffSuplier;
    partner?: string; // есть у самолетных тарифов
}

interface ISegmentTariffPartnerOrderRequest {
    httpMethod: EHttpMethod.GET | EHttpMethod.POST;
    params: Record<string, string>;
    url: string;
}

interface ISegmentTariffClassFromBackend {
    price: IPrice;
    orderUrl: string;
    seats?: number; // количество мест
    severalPrices?: boolean;
    trainOrderUrl?: string; // есть в поездатых тарифах от trainApi
    trainOrderUrlOwner?: EOrderUrlOwner | null; // есть в поездатых тарифах
    partnerOrderRequest?: ISegmentTariffPartnerOrderRequest; // для ссылок на автобусы
    deepUrl?: string;
}

interface ISubSegmentFromBackend {
    title: string;
    transport: ISegmentTransport;
    company: ISegmentCompany;
    arrival: TInstantDateTime;
    arrivalEvent: string | null;
    arrivalEventKey: string | null;
    departure: TInstantDateTime;
    departureEvent: string | null;
    departureEventKey: string | null;
    duration: number;
    isInterval: false;
    isThroughTrain: boolean;
    number: string;
    startDate: string;
    stationFrom: ISegmentStation;
    stationTo: ISegmentStation;
    stops: string;
    suburbanFacilities: null;
    thread: ISegmentThread | null;
    arrivalLocalDt: TInstantDateTime; // Время прибытия в таймзоне пункта прибытия
    departureLocalDt: TInstantDateTime; // Время отправления в таймзоне пункта отправления

    isFuzzyFrom?: boolean; // признак, того, что время отправления не точное
    isFuzzyTo?: boolean; // признак, того, что время прибытия не точное
}

interface ISegmentDaysByTimezone {
    [timezone: string]: {
        text: string; // Строка, описывающая дни хождения для поиска на все дни. Например "ежедневно"
        exceptText?: string; // Строка, описывающая исключения в днях хождения
        shortText?: string; // Судя по коду: в котором это используется, это строка с содержимым наподобие свойства "text"
        schedulePlanAppendix?: string; //  Судя по коду: в котором это используется, это незначительное по важности дополнение к полю "shortText"
    };
}

interface IMaskDaysOfWalkingMonth {
    [month: string]: number[];
}

interface IMaskDaysOfWalking {
    [year: string]: IMaskDaysOfWalkingMonth;
}

/**
 * Описывает данные о кодшерных рейсах.
 * Кодшер - авиарейс другой компании, который по факту летит на самолете основной компании-перевозчика.
 * Допустим самолет Аэрофлота, но так же на этом же самолете летит рейс S7, который повторяет маршрут Аэрофлота, но
 * может быть дешевле/дороже рейса Аэрофлота.
 */
interface ISegmentCodeshare {
    company: ISegmentCompany;
    number: string;
    tariffsKeys: string[];
}

export interface ISegmentFromBackend {
    title: string;
    arrival: TInstantDateTime;
    arrivalEvent: ISegmentEvent | null;
    arrivalEventKey: string | null;
    departure: TInstantDateTime;
    departureEvent: ISegmentEvent | null;
    departureEventKey: string | null;
    duration: number; // Продолжительность поездки в секундах
    company: ISegmentCompany | null; // null часто встречается у автобусных сегментов, когда нет данных о перевозчике
    isInterval: boolean;
    isThroughTrain: boolean | null;
    number: string;
    stationFrom: ISegmentStation;
    stationTo: ISegmentStation;
    transport: ISegmentTransport;
    suburbanFacilities: ISegmentFacility[] | null;
    arrivalLocalDt: TInstantDateTime; // Время прибытия в таймзоне пункта прибытия
    departureLocalDt: TInstantDateTime; // Время отправления в таймзоне пункта отправления

    thread?: ISegmentThread | null;
    tariffs?: ISegmentTariffs<ISegmentTariffClassFromBackend>;
    tariffsKeys?: string[];
    key?: string;
    keys?: string[];
    stops?: string; // информация об остановках. Например "кроме: Фрязино-Тов."
    source?: string; // возможно такое свойство есть у автобусных сегментов
    ufsTitle?: string;
    twoStorey?: boolean;
    rawTrainCategory?: string;
    hasDynamicPricing?: boolean; // приходит с тарифами поездов
    isSuburban?: boolean;
    originalNumber?: string;
    rawTrainName?: null | string;
    hasTrainTariffs?: boolean | null;
    startDate?: string;
    maxArrival?: string; // минимальное время прибытия. Есть у мета-сегментов
    minArrival?: string; // максимальное время прибытия. Есть у мета-сегментов
    subSegments?: ISubSegmentFromBackend[]; // сегменты мета-сегмента
    canSupplySegments?: boolean /* означает, что данные сегмента с тарифом можно использовать
    как рейс-пополнение (т.е. добавить в выдачу, как самостоятельный сегмент) */;
    companies?: ISegmentCompany[]; // информация о компаниях для мета-сегментов
    url?: string; // ссылка на нитку у самолетных сегментов
    isFuzzyFrom?: boolean; // признак, того, что время отправления не точное
    isFuzzyTo?: boolean; // признак, того, что время прибытия не точное
    daysByTimezone?: ISegmentDaysByTimezone; // Тексты, опсывающие дни хождения для поиска на все дни
    runDays?: IMaskDaysOfWalking; // Макса дней хождения для поиска на все дни
    canPayOffline?: boolean; // Встречается в автобусных тарифах
    yBusId?: string; // Встречается в автобусных тарифах
    intervalThreadDepartureFromDate?: TDate; // Дата старта нитки для интервальных рейсов. Нужна для того, чтобы строить корректные ссылки на страницу нитки со страницы поиска
    codeshares?: ISegmentCodeshare[]; // Список кодшерных рейсов, которые летят на донном борту
    trainPurchaseNumbers?: string[]; // Для сегмента электричек ЦППК, которые являются одновременно и поездами, номера поездовых ниток.
    salesLimitInDays?: number; // За сколько дней открывается продажа билетов на поезда. Бывает для поездов на выдаче на все дни.
}
