// -------------------------------------------------
// Validation
// -------------------------------------------------
import {IFieldValidation, TFieldGroupId} from './form';

export enum EValidationType {
    REQUIRED = 'required',
    AVAILABLE = 'available',

    REGEX = 'regex',
    MAX_LENGTH = 'maxLength',
    MIN_LENGTH = 'minLength',

    MAX_VALUE = 'maxValue',
    MIN_VALUE = 'minValue',

    ONE_OF = 'oneOf',

    MAX_DATE = 'maxDate',
    MIN_DATE = 'minDate',
    MAX_DATE_FROM_TODAY = 'maxDateFromToday',
    MIN_DATE_FROM_TODAY = 'minDateFromToday',
    MAX_DATE_FROM_FIELD = 'maxDateFromField',
    MIN_DATE_FROM_FIELD = 'minDateFromField',
}

type TOneOfVales = string[];

type TDateOffsetScale = 'year' | 'month' | 'day';

interface IDateOffset {
    offset: number;
    scale: TDateOffsetScale;
}

interface IDateOffsetFromField extends IDateOffset {
    field: IFieldPath;
}

export type TValidationParams =
    | number
    | string
    | boolean
    | TOneOfVales
    | IDateOffset
    | IDateOffsetFromField;

/** Общий интерфейс для наследования типов валидации */
interface IBaseValue<T extends EValidationType, P extends TValidationParams> {
    type: T;
    params: P;
}

interface IWithError {
    errorMessage: string;
}

/* Common validations */

interface IRequiredValidation
    extends IBaseValue<EValidationType.REQUIRED, boolean> {}
interface IAvailableValidation
    extends IBaseValue<EValidationType.AVAILABLE, boolean> {}

export type TCommonValue = IRequiredValidation | IAvailableValidation;
export type TCommonValidation = TCommonValue & IWithError;

/* String validations */

interface IRegexValidation extends IBaseValue<EValidationType.REGEX, string> {}
interface IMaxLengthValidation
    extends IBaseValue<EValidationType.MAX_LENGTH, number> {}
interface IMinLengthValidation
    extends IBaseValue<EValidationType.MIN_LENGTH, number> {}

export type TStringValue =
    | TCommonValidation
    | IRegexValidation
    | IMaxLengthValidation
    | IMinLengthValidation;

export type TStringValidation = TStringValue & IWithError;

/* Number validations */

interface IMaxValueValidation
    extends IBaseValue<EValidationType.MAX_VALUE, number> {}
interface IMinValueValidation
    extends IBaseValue<EValidationType.MIN_VALUE, number> {}

export type TNumberValue =
    | TCommonValidation
    | IMaxValueValidation
    | IMinValueValidation;

export type TNumberValidation = TNumberValue & IWithError;

/* Select validations */

interface IOneOfValuesValidation
    extends IBaseValue<EValidationType.ONE_OF, TOneOfVales> {}

export type TSelectValue = TCommonValidation | IOneOfValuesValidation;
export type TSelectValidation = TSelectValue & IWithError;

/* Date validations */

/** Строка в стандартизированном формате, например '22.01.2010' */
interface IMaxDateValidation
    extends IBaseValue<EValidationType.MAX_DATE, string> {}
interface IMinDateValidation
    extends IBaseValue<EValidationType.MIN_DATE, string> {}

interface IMaxDateFromTodayValidation
    extends IBaseValue<EValidationType.MAX_DATE_FROM_TODAY, IDateOffset> {}
interface IMinDateFromTodayValidation
    extends IBaseValue<EValidationType.MIN_DATE_FROM_TODAY, IDateOffset> {}

interface IMaxDateFromFieldValidation
    extends IBaseValue<
        EValidationType.MAX_DATE_FROM_FIELD,
        IDateOffsetFromField
    > {}
interface IMinDateFromFieldValidation
    extends IBaseValue<
        EValidationType.MIN_DATE_FROM_FIELD,
        IDateOffsetFromField
    > {}

export type TDateValue =
    | TCommonValidation
    | IMaxDateValidation
    | IMinDateValidation
    | IMaxDateFromTodayValidation
    | IMinDateFromTodayValidation
    | IMaxDateFromFieldValidation
    | IMinDateFromFieldValidation;

export type TDateValidation = TDateValue & IWithError;

/** Main validation type */
export type TValidations =
    | TStringValidation[]
    | TNumberValidation[]
    | TDateValidation[]
    | TSelectValidation[];

// -------------------------------------------------
// Dependent validation
// -------------------------------------------------

export type TDependentValue =
    | TStringValue[]
    | TNumberValue[]
    | TDateValue[]
    | TSelectValue[];

interface IFieldPath {
    fieldGroupId: TFieldGroupId;
    fieldName: string;

    /**
     * Когда есть зависимость от массива сущностей
     * Например: Хотя бы у одного пассажира документ - Паспорт РФ
     * Например: Все пассажиры дети
     */
    type?: 'every' | 'some';
}

interface IDependencyCondition {
    path: IFieldPath;
    value: TDependentValue;
}

/**
 * Валидация соотвествующая комбинации значений полей.
 *
 * При созданни пассажира в аккаунте, телефон будет обязательными, если:
 * 1) поставить галку  о согласии на рассылку (agreement.notification.value === true),
 * 2) при пустом email (contacts.email.value === '')
 */
export interface IDependentValidation {
    conditions: IDependencyCondition[];
    validation: IFieldValidation<TValidations>;
}
