import {IHotelsSearchFormWidget} from 'widgets/HotelsSearchForm/HotelsSearchForm';
import {IWidgetConstructor} from 'widgets/WidgetConstructor/WidgetConstructor';

import {IWidget} from 'types/widgets/IWidget';
import {IMetrika} from 'types/metrika/IMetrika';
import {EWidgetType} from 'types/widgets/EWidgetType';
import {IWidgetDeps} from 'types/widgets/IWidgetDeps';
import {IBuildOptions} from 'types/widgets/IBuildOptions';
import {IYaTravelAffiliate} from 'types/IYaTravelAffiliate';
import {IErrorBooster} from 'types/errorBooster/IErrorBooster';

import Metrika from 'utilities/metrika/Metrika';
import {WidgetMetrika} from 'utilities/metrika/WidgetMetrika';
import ErrorBooster from 'utilities/errorBooster/ErrorBooster';

class YaTravelAffiliate implements IYaTravelAffiliate {
    static readonly namespace: string = 'YaTravelAffiliate';
    static _instance?: YaTravelAffiliate;
    private metrika: IMetrika = new Metrika();
    private errorBooster: IErrorBooster = new ErrorBooster();

    constructor() {
        if (YaTravelAffiliate._instance) {
            return YaTravelAffiliate._instance;
        }

        this._init();
    }

    _init(): void {
        this._initServiceFrame();

        // Таймаут выставлен т.к. dispatchEvent работает синхронно =>
        // если будет подписка на это событие, в котором идёт обращение к YaTravelAffiliate,
        // то мы получим ошибку, т.к. глобальная переменная на этот момент ещё не инициализирована
        setTimeout(() =>
            dispatchEvent(
                new CustomEvent(`${YaTravelAffiliate.namespace}Loaded`),
            ),
        );
    }

    async createWidget(options: IBuildOptions): Promise<IWidget> {
        switch (options.type) {
            case EWidgetType.HOTELS_SEARCH_FORM:
                return import('widgets/HotelsSearchForm/HotelsSearchForm').then(
                    ({default: HotelsSearchFormWidget}) =>
                        new HotelsSearchFormWidget(
                            options as IBuildOptions<IHotelsSearchFormWidget>,
                            this._getDeps(options),
                        ),
                );
            case EWidgetType.CONSTRUCTOR:
                return import(
                    'widgets/WidgetConstructor/WidgetConstructor'
                ).then(
                    ({default: WidgetConstructor}) =>
                        new WidgetConstructor(
                            options as IBuildOptions<IWidgetConstructor>,
                            this._getDeps(options),
                        ),
                );
        }
    }

    private _initServiceFrame(): void {
        if (document.body) {
            this._injectServiceFrame();
        } else {
            window.addEventListener(
                'DOMContentLoaded',
                this._injectServiceFrame.bind(this),
                {once: true},
            );
        }
    }

    private _injectServiceFrame(): void {
        const iframe = document.createElement('iframe');

        iframe.src = `${__HOSTNAME__ || ''}/service.html`;
        iframe.style.position = 'absolute';
        iframe.style.top = '-9999px';
        iframe.style.left = '-9999px';

        document.body.appendChild(iframe);
    }

    private _getDeps(options: IBuildOptions): IWidgetDeps {
        return {
            metrika: new WidgetMetrika(this.metrika, {
                type: options.type,
                affiliate: options.affiliate,
            }),
            errorBooster: this.errorBooster,
        };
    }
}

export default YaTravelAffiliate;
