import React, {useMemo} from 'react';
import classNames from 'classnames';

import {IWithClassName} from 'types/IWithClassName';

import {proxyRefCb} from 'utilities/proxyRefCb';

import {ButtonPresenter} from './controls/ButtonPresenter';

export const DEFAULT_SIZE = 'm-inset';

const iconSizeMap = {
    s: 12,
    m: 16,
    'm-inset': 16,
    l: 16,
    xl: 24,
};

export function getScaledIcon(
    icon: JSX.Element | undefined,
    size: TButtonSize | undefined,
    onlyIcon: Boolean | true,
    type?: 'left' | 'right',
): React.ReactNode {
    if (!icon) {
        return;
    }

    const scaledSize = (size && iconSizeMap[size]) || iconSizeMap['m'];
    const iconClassName = classNames(
        type ? `YTButton-Icon_${type}` : undefined,
        onlyIcon ? 'YTButton-Icon_only' : undefined,
        icon.props.className,
    );

    return React.cloneElement(
        icon,
        {
            className: iconClassName,
            width: icon.props.width || scaledSize,
            height: icon.props.height || scaledSize,
        },
        [],
    );
}

export type TButtonSize = 's' | 'm-inset' | 'm' | 'l' | 'xl';
export type TButtonTheme =
    | 'secondary'
    | 'primary'
    | 'outlined'
    | 'attention'
    | 'raised'
    | 'plain'
    | 'plus';

export interface IButtonProps extends IWithClassName {
    value?: string;

    /**
     * Тема кнопки
     *
     * @default 'secondary'
     */
    theme?: TButtonTheme;

    children?: React.ReactNode;

    type?: 'button' | 'submit' | 'reset';

    /**
     * Ширина кнопки
     *
     * @default 'auto'
     */
    width?: 'auto' | 'max';

    disabled?: boolean;

    icon?: JSX.Element;

    name?: string;

    id?: string;

    as?: React.ElementType;

    tabIndex?: number;

    /* Функции для render иконок(position: absolute;) */

    iconLeft?: JSX.Element;

    iconRight?: JSX.Element;

    /**
     * Размер кнопки
     *
     * @default 'm-inset'
     */
    size?: TButtonSize;

    state?: 'error';

    shape?: 'circle';

    innerRef?: React.Ref<HTMLElement>;

    onClick?: React.MouseEventHandler<HTMLElement>;

    onMouseEnter?: React.MouseEventHandler<HTMLElement>;

    onMouseLeave?: React.MouseEventHandler<HTMLElement>;

    onMouseOver?: React.MouseEventHandler<HTMLElement>;

    onFocus?: React.FocusEventHandler<HTMLElement>;

    onBlur?: React.FocusEventHandler<HTMLElement>;

    onKeyDown?: React.KeyboardEventHandler<HTMLElement>;
}

const Button: React.FC<IButtonProps> = ({
    name,
    id,
    as,
    tabIndex,
    size = DEFAULT_SIZE,
    width = 'auto',
    theme = 'secondary',
    state,
    shape,
    children,
    disabled,
    icon,
    type,
    value,
    className,
    innerRef,
    onClick,
    onBlur,
    onFocus,
    onMouseEnter,
    iconLeft,
    iconRight,
    onMouseLeave,
    onMouseOver,
    onKeyDown,
}) => {
    const leftIcon = useMemo(() => {
        return (
            (icon || iconLeft) &&
            getScaledIcon(icon || iconLeft, size, !children, 'left')
        );
    }, [icon, iconLeft, size, children]);

    const rightIcon = useMemo(() => {
        return iconRight && getScaledIcon(iconRight, size, !children, 'right');
    }, [iconRight, size, children]);

    const buttonContent = useMemo(() => {
        return (
            <span className={classNames('YTButton-Text')}>
                {leftIcon}
                {children}
                {rightIcon}
            </span>
        );
    }, [children, rightIcon, leftIcon]);

    const commonProps = {
        innerRef:
            typeof innerRef === 'function' ? proxyRefCb(innerRef) : innerRef,
        name,
        id,
        as,
        tabIndex,
        children: buttonContent,
        size,
        state,
        shape,
        value,
        className,
        disabled,
        view: 'default' as 'default',
        width: width === 'max' ? 'max' : undefined,
        theme,
        type,
        onClick,
        onBlur,
        onFocus,
        onMouseEnter,
        onMouseLeave,
        onMouseOver,
        onKeyDown,
    };

    // @ts-ignore
    return <ButtonPresenter {...commonProps} />;
};

export default Button;
