import React, {useCallback, useMemo, useState} from 'react';
import {noop} from 'lodash';

import {
    ECalendarScrollSource,
    ECalendarType,
    TCalendarPricesMask,
} from 'components/Calendar/types';
import {IWithClassName} from 'types/IWithClassName';
import {TCalendarPrices} from 'types/common/calendarPrice/ICalendarPrice';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import createCalendarMonths from 'components/Calendar/utilities/createCalendarMonths';
import createCalendarWeekdays from 'components/Calendar/utilities/createCalendarWeekdays';

import CalendarMonthsList from './components/CalendarMonthsList/CalendarMonthsList';
import CalendarMonthsGrid from './components/CalendarMonthsGrid/CalendarMonthsGrid';
import calendarScrollContainerHOC from './components/CalendarScrollContainer/CalendarScrollContainer';

import cx from './Calendar.scss';

export interface ICalendarProps extends IWithClassName {
    calendarWrapperClassName?: string;
    dayClassName?: string;
    weekdaysClassName?: string;
    monthsGridClassName?: string;

    minDate: Date;
    maxDate: Date;
    nowDate: Date;
    startDate?: Nullable<Date>;
    endDate?: Nullable<Date>;
    calendarType?: ECalendarType;
    weekStart?: number;
    siblingMonths?: boolean;
    headerNode?: React.ReactNode;
    footerNode?: React.ReactNode;
    isRenderMonthsList: boolean;
    lastScrollSource: Nullable<ECalendarScrollSource>;
    withAnimation?: boolean;
    scrollPartPosition: number;
    cutPastWeeks?: boolean;
    canScrollIntoSelectedDate?: boolean;
    prices?: TCalendarPrices;
    mask?: TCalendarPricesMask;

    onScroll?: (scrollParams: {
        scrollPartPosition: 0;
        source: Nullable<ECalendarScrollSource>;
    }) => void;
    onDaySelect?: (data: {
        calendarType: ECalendarType;
        selectedDate: Date;
    }) => void;

    onMonthsGridTouchStart?: (event: TouchEvent) => void;
    onMonthsGridTouchMove?: (event: TouchEvent) => void;
    onMonthsGridTouchEnd?: (event: TouchEvent) => void;
}

const Calendar: React.FC<ICalendarProps> = props => {
    const {
        className,
        calendarWrapperClassName,
        dayClassName,
        weekdaysClassName,
        monthsGridClassName,
        mask,
        prices,
        minDate,
        maxDate,
        nowDate,
        onDaySelect = noop,
        onScroll = noop,
        startDate = null,
        endDate = null,
        calendarType = ECalendarType.START_DATE,
        weekStart = 1,
        siblingMonths = false,
        headerNode = null,
        footerNode = null,
        isRenderMonthsList = true,
        lastScrollSource = '',
        scrollPartPosition = 0,
        cutPastWeeks = false,
        canScrollIntoSelectedDate = true,
        onMonthsGridTouchEnd,
        onMonthsGridTouchStart,
        onMonthsGridTouchMove,
    } = props;
    const deviceType = useDeviceType();
    const [hoveredDate, setHoveredDate] = useState<false | Date>(false);
    const handleDayClick = useCallback(
        (selectedDate: Date) => {
            onDaySelect({calendarType, selectedDate});
        },
        [calendarType, onDaySelect],
    );

    const handleDayMouseLeave = useCallback(() => {
        if (deviceType.isTouch) {
            return;
        }

        setHoveredDate(false);
    }, [setHoveredDate, deviceType.isTouch]);

    const handleDayMouseEnter = useCallback(
        (newValue: Date) => {
            if (deviceType.isTouch) {
                return;
            }

            setHoveredDate(newValue);
        },
        [setHoveredDate, deviceType.isTouch],
    );

    const months = useMemo(
        () =>
            createCalendarMonths({
                minDate,
                maxDate,
                weekStart,
                siblingMonths,
                cutPastWeeks,
                cutDate: nowDate,
            }),
        [minDate, maxDate, weekStart, siblingMonths, cutPastWeeks, nowDate],
    );
    const weekdays = useMemo(
        () => createCalendarWeekdays({weekStart}),
        [weekStart],
    );

    const isRangeSelected = useMemo(() => {
        if (!startDate) {
            return false;
        }

        if (endDate && Number(startDate) !== Number(endDate)) {
            return true;
        }

        return (
            calendarType === ECalendarType.END_DATE &&
            hoveredDate &&
            Number(hoveredDate) !== Number(startDate)
        );
    }, [startDate, endDate, hoveredDate, calendarType]);

    return (
        <div
            className={cx(
                'calendar',
                className,
                deviceMods('calendar', deviceType),
            )}
        >
            {isRenderMonthsList && (
                <CalendarMonthsList
                    className={cx('monthsList')}
                    months={months}
                    onChangeScrollPosition={onScroll}
                    scrollPartPosition={scrollPartPosition}
                />
            )}
            <div className={cx('calendarWrapper', calendarWrapperClassName)}>
                {headerNode && <div className={cx('header')}>{headerNode}</div>}
                <CalendarMonthsGrid
                    className={cx('monthsGrid', monthsGridClassName)}
                    dayClassName={dayClassName}
                    weekdaysClassName={weekdaysClassName}
                    months={months}
                    weekdays={weekdays}
                    startDate={startDate ? startDate : undefined}
                    endDate={endDate ? endDate : undefined}
                    nowDate={nowDate}
                    minDate={minDate}
                    maxDate={maxDate}
                    isRangeSelected={isRangeSelected}
                    mask={mask}
                    deviceType={deviceType}
                    hoveredDate={hoveredDate ? hoveredDate : undefined}
                    calendarType={calendarType}
                    onChangeScrollPosition={onScroll}
                    lastScrollSource={lastScrollSource}
                    scrollPartPosition={scrollPartPosition}
                    canScrollIntoSelectedDate={canScrollIntoSelectedDate}
                    prices={prices}
                    onDayClick={handleDayClick}
                    onDayMouseLeave={handleDayMouseLeave}
                    onDayMouseEnter={handleDayMouseEnter}
                    onTouchStart={onMonthsGridTouchStart}
                    onTouchMove={onMonthsGridTouchMove}
                    onTouchEnd={onMonthsGridTouchEnd}
                />
                {footerNode}
            </div>
        </div>
    );
};

export default calendarScrollContainerHOC(Calendar);
