import React, {MouseEventHandler, PureComponent} from 'react';
import _noop from 'lodash/noop';

import {ICalendarDay} from 'components/Calendar/types';
import {IWithClassName} from 'types/IWithClassName';
import {
    ICalendarEmptyPrice,
    ICalendarPrice,
} from 'types/common/calendarPrice/ICalendarPrice';
import {IDevice} from 'types/IDevice';

import getUTCDateByParams from '../../utilities/getUTCDateByParams';
import {deviceMods} from 'utilities/stylesUtils';
import getCalendarDayHeight from 'components/Calendar/utilities/getCalendarDayHeight';

import CalendarPriceDay from 'components/Calendar/components/CalendarPriceDay/CalendarPriceDay';

import cx from './CalendarDayNew.scss';

interface ICalendarDayProps extends IWithClassName {
    day: ICalendarDay | false;
    isRangeSelected: boolean;
    deviceType: IDevice;

    setScrolledDayNode: Function;
    onDayClick: (selectedDate: Date, dayNode: HTMLDivElement) => void;
    onDayMouseLeave: (hoveredDate: Date) => void;
    onDayMouseEnter: (hoveredDate: Date) => void;
    price?: ICalendarPrice | ICalendarEmptyPrice;
    setActiveDayRef: Function;

    isHovered: boolean;
    isStartDate: boolean;
    isEndDate: boolean;
    isScrolledDate: boolean;
    isRangeDay: boolean;
    isWeekend: boolean;
    isActive: boolean;
    isDisable: boolean;
}

class CalendarDayNew extends PureComponent<ICalendarDayProps> {
    private rootNode: HTMLDivElement | null = null;

    static defaultProps = {
        className: '',

        onDayClick: _noop,
        onDayMouseLeave: _noop,
        onDayMouseEnter: _noop,
        setScrolledDayNode: _noop,
        setActiveDayRef: _noop,

        isHovered: false,
        isStartDate: false,
        isEndDate: false,
        isScrolledDate: false,
        isRangeDay: false,
        isWeekend: false,
        isActive: false,
        isDisable: false,
    };

    componentDidMount(): void {
        this.setSelectedDayNode();
    }

    componentDidUpdate(prevProps: ICalendarDayProps): void {
        const {setActiveDayRef, isStartDate, isEndDate} = this.props;

        if (
            (prevProps.isStartDate !== isStartDate && isStartDate) ||
            (prevProps.isEndDate !== isEndDate && isEndDate)
        ) {
            setActiveDayRef(this.getRootNodeRef());
        }
    }

    setSelectedDayNode(): void {
        const {setScrolledDayNode, isScrolledDate} = this.props;

        if (isScrolledDate) {
            const dayNode = this.getRootNodeRef();

            setScrolledDayNode({
                dayNode,
            });
        }
    }

    setRootNodeRef = (rootNode: HTMLDivElement): void => {
        this.rootNode = rootNode;
    };

    getRootNodeRef = (): HTMLDivElement | null => {
        return this.rootNode;
    };

    getDayParams(): Date {
        const {day} = this.props;

        // as: тк вызывается только при hover на существующий день
        // (непустая клетка календаря в таблице месяце)
        return getUTCDateByParams(day as ICalendarDay);
    }

    handleMouseLeave: MouseEventHandler = () => {
        const {onDayMouseLeave} = this.props;
        const day = this.getDayParams();

        onDayMouseLeave(day);
    };

    handleMouseEnter: MouseEventHandler = () => {
        const {onDayMouseEnter} = this.props;
        const day = this.getDayParams();

        onDayMouseEnter(day);
    };

    handleClick: MouseEventHandler<HTMLDivElement> = e => {
        const {onDayClick} = this.props;
        const day = this.getDayParams();

        onDayClick(day, e.currentTarget);
    };

    renderEmptyDay(): React.ReactNode {
        const {className, deviceType} = this.props;

        return (
            <div
                className={cx(
                    'day',
                    'day_empty_yes',
                    deviceMods('day', deviceType),
                    className,
                )}
            />
        );
    }

    renderDay({day}: ICalendarDay): React.ReactNode {
        const {
            className,
            deviceType,
            isWeekend,
            isHovered,
            isStartDate,
            isEndDate,
            isRangeDay,
            isActive,
            isDisable,
            isRangeSelected,
        } = this.props;

        return (
            <div
                ref={this.setRootNodeRef}
                className={cx(
                    'day',
                    {
                        day_type_weekend: isWeekend,
                        day_type_hover: isHovered,
                        day_type_start: isStartDate,
                        day_type_startWithContinue: isRangeSelected,
                        day_type_end: isEndDate,
                        day_type_inRange: isRangeDay,
                        day_type_active: isActive,
                        day_type_disabled: isDisable,
                    },
                    deviceMods('day', deviceType),
                    className,
                )}
                onMouseEnter={this.handleMouseEnter}
                onMouseLeave={this.handleMouseLeave}
                style={{height: getCalendarDayHeight(deviceType.isMobile)}}
                onClick={this.handleClick}
            >
                <span>{day}</span>
                {this.renderPrice()}
            </div>
        );
    }

    renderPrice(): React.ReactNode {
        const {isDisable, price} = this.props;

        return <CalendarPriceDay price={price} isDisable={isDisable} />;
    }

    render(): React.ReactNode {
        const {day} = this.props;

        return day ? this.renderDay(day) : this.renderEmptyDay();
    }
}

export default CalendarDayNew;
