import React, {PureComponent} from 'react';
import _noop from 'lodash/noop';

import {IWithClassName} from 'types/IWithClassName';
import {IWithDeviceType} from 'types/IWithDeviceType';
import {
    ECalendarType,
    ICalendarDay,
    ICalendarMonth,
    TCalendarMask,
} from 'components/Calendar/types';
import {
    ICalendarPrice,
    TCalendarPrices,
    ICalendarEmptyPrice,
    ECalendarEmptyPriceReason,
} from 'types/common/calendarPrice/ICalendarPrice';

import getMonthWeeks from '../../utilities/getMonthWeeks';
import getCalendarDaySettings, {
    TCalendarDaySettings,
} from '../../utilities/getCalendarDaySettings';
import getISObyDate from '../../utilities/getISObyDate';
import getUTCDateByParams from '../../utilities/getUTCDateByParams';
import {canMoveMonthLabel} from 'components/Calendar/utilities/canMoveMonthLabel';

import CalendarDay from '../../components/CalendarDay/CalendarDay';

import cx from './CalendarMonth.scss';

interface ICalendarMonthProps extends IWithClassName, IWithDeviceType {
    dayClassName: string;
    month: ICalendarMonth;
    nowDate: Date;
    minDate: Date;
    maxDate: Date;
    startDate: Date;
    endDate: Date;
    hoveredDate: Date;
    isRangeSelected: boolean;
    showLabel: boolean;
    calendarType: ECalendarType;
    setActiveDayRef?: Function;
    setScrolledDayNode: Function;
    prices?: TCalendarPrices;
    mask?: TCalendarMask;
    onDayClick: Function;
    onDayMouseLeave: Function;
    onDayMouseEnter: Function;
}

class CalendarMonth extends PureComponent<ICalendarMonthProps> {
    static readonly defaultProps = {
        className: '',
        dayClassName: '',
        month: {},
        startDate: null,
        endDate: null,
        hoveredDate: false,
        showLabel: true,
        calendarType: ECalendarType.START_DATE,
        setScrolledDayNode: _noop,
        onDayClick: _noop,
        onDayMouseLeave: _noop,
        onDayMouseEnter: _noop,
    };

    private getPriceForDay(
        day: ICalendarDay | false,
    ): ICalendarPrice | ICalendarEmptyPrice | undefined {
        const {prices} = this.props;

        if (!prices || !day) {
            return undefined;
        }

        const price = prices[getISObyDate(getUTCDateByParams(day))];

        if (!price) {
            return {emptyPriceReason: ECalendarEmptyPriceReason.OTHER};
        }

        return price;
    }

    private getDaySettings(day: ICalendarDay | false): TCalendarDaySettings {
        const {
            mask,
            startDate,
            endDate,
            nowDate,
            minDate,
            maxDate,
            hoveredDate,
            calendarType,
        } = this.props;

        return getCalendarDaySettings({
            day,
            mask,
            startDate,
            endDate,
            nowDate,
            minDate,
            maxDate,
            hoveredDate,
            calendarType,
        });
    }

    private renderMonthDay = (
        day: false | ICalendarDay,
        dayIndex: number,
    ): React.ReactNode => {
        const {
            deviceType,
            dayClassName,
            isRangeSelected,
            onDayClick,
            onDayMouseLeave,
            onDayMouseEnter,
            setActiveDayRef,
            setScrolledDayNode,
        } = this.props;

        const daySettings = this.getDaySettings(day);

        return (
            <CalendarDay
                className={dayClassName}
                deviceType={deviceType}
                day={day}
                key={dayIndex}
                price={this.getPriceForDay(day)}
                isRangeSelected={isRangeSelected}
                onDayClick={onDayClick}
                onDayMouseLeave={onDayMouseLeave}
                onDayMouseEnter={onDayMouseEnter}
                setScrolledDayNode={setScrolledDayNode}
                setActiveDayRef={setActiveDayRef}
                {...daySettings}
            />
        );
    };

    private renderMonthWeek = (
        week: (false | ICalendarDay)[],
        weekIndex: number,
    ): React.ReactNode => {
        return (
            <div className={cx('week')} key={weekIndex}>
                {week.map(this.renderMonthDay)}
            </div>
        );
    };

    private renderMonthLabel(): React.ReactNode {
        const {month, showLabel} = this.props;

        if (!showLabel) {
            return null;
        }

        const isMovedMonth = canMoveMonthLabel(month);

        return (
            <div
                className={cx('month', {
                    month_moved_yes: isMovedMonth,
                })}
            >
                {month.monthLabel}
            </div>
        );
    }

    private renderMonthDays = (): React.ReactNode => {
        const {
            month: {monthDays},
        } = this.props;
        const monthWeeks = getMonthWeeks(monthDays);

        return monthWeeks.map(this.renderMonthWeek);
    };

    render(): React.ReactNode {
        const {className} = this.props;

        return (
            <div className={cx('calendarMonth', className)}>
                {this.renderMonthLabel()}
                {this.renderMonthDays()}
            </div>
        );
    }
}

export default CalendarMonth;
