import React, {PureComponent} from 'react';
import {Spring, animated} from 'react-spring';
import _noop from 'lodash/noop';

import {IWithClassName} from 'types/IWithClassName';
import {ICalendarMonth} from 'components/Calendar/types';

import cx from './CalendarMonthsList.scss';

const ACTIVE_MONTHS_SCROLL_HEIGHT = 40;
const ACTIVE_MONTHS_SCROLL_OFFSET = 15;

interface ICalendarMonthsListProps extends IWithClassName {
    scrollPartPosition: number;
    onChangeScrollPosition: Function;
    months: ICalendarMonth[];
}

interface ICalendarMonthsListState {
    monthsListHeight: number;
}

type TMonthsListNode = HTMLDivElement | null;

class CalendarMonthsList extends PureComponent<
    ICalendarMonthsListProps,
    ICalendarMonthsListState
> {
    private monthsListNode: TMonthsListNode = null;

    static readonly defaultProps: ICalendarMonthsListProps = {
        className: '',
        scrollPartPosition: 0,
        onChangeScrollPosition: _noop,
        months: [],
    };

    constructor(props: ICalendarMonthsListProps) {
        super(props);

        this.state = {
            monthsListHeight: 0,
        };
    }

    componentDidMount(): void {
        this.calculateAndSetMonthsListHeight();
    }

    private calculateAndSetMonthsListHeight(): void {
        const monthsListNode = this.getMonthsListRef();

        if (!monthsListNode) {
            return;
        }

        const {height: monthsListHeight} =
            monthsListNode.getBoundingClientRect();

        this.setState({monthsListHeight});
    }

    private getMonthsScrollPositionProps(): number {
        const {monthsListHeight} = this.state;
        const {scrollPartPosition} = this.props;

        return (
            (monthsListHeight - ACTIVE_MONTHS_SCROLL_HEIGHT) *
            scrollPartPosition
        );
    }

    private handleMonthClick(monthIndex: number): void {
        const {onChangeScrollPosition, months} = this.props;
        const allMonthCount = months.length;
        const scrollPartPosition = monthIndex / (allMonthCount - 1);

        onChangeScrollPosition({scrollPartPosition, source: 'monthsList'});
    }

    private setMonthsListRef = (monthsListNode: TMonthsListNode): void => {
        this.monthsListNode = monthsListNode;
    };

    private getMonthsListRef = (): TMonthsListNode => {
        return this.monthsListNode;
    };

    private renderActiveMonthsScrollPosition(): React.ReactNode {
        const scrollTopPosition = this.getMonthsScrollPositionProps();

        return (
            <Spring to={{transform: `translateY(${scrollTopPosition}px)`}}>
                {({transform}): React.ReactElement => (
                    <animated.div
                        className={cx('activeMonthsScroll')}
                        style={{
                            top: ACTIVE_MONTHS_SCROLL_OFFSET,
                            height: ACTIVE_MONTHS_SCROLL_HEIGHT,
                            transform,
                        }}
                    />
                )}
            </Spring>
        );
    }

    private renderMonth = (
        {monthLabel, month, year}: ICalendarMonth,
        monthIndex: number,
    ): React.ReactNode => {
        const isFirstMonth = month === 0;

        return (
            <div
                key={monthIndex}
                className={cx('month')}
                onClick={this.handleMonthClick.bind(this, monthIndex)}
            >
                {monthLabel}
                {isFirstMonth && ` ${year}`}
            </div>
        );
    };

    private renderMonthsList(): React.ReactNode {
        const {months} = this.props;

        return (
            <div ref={this.setMonthsListRef} className={cx('monthsList')}>
                {months.map(this.renderMonth)}
            </div>
        );
    }

    render(): React.ReactNode {
        const {className} = this.props;

        return (
            <div className={cx('calendarMonthsList', className)}>
                {this.renderActiveMonthsScrollPosition()}
                {this.renderMonthsList()}
            </div>
        );
    }
}

export default CalendarMonthsList;
