import {FC} from 'react';

import {CURRENCY_RUB} from 'constants/currencyCodes';
import {CHAR_WIDE_MIDDLE_DASH} from 'constants/charCodes';

import {
    ECalendarEmptyPriceReason,
    ICalendarEmptyPrice,
    ICalendarPrice,
} from 'types/common/calendarPrice/ICalendarPrice';

import {isCalendarDayWithPrice} from './utilities/isCalendarDayWithPrice';

import * as i18nPriceCalendarBlock from 'i18n/components-price-calendar';

import SearchIcon from 'icons/12/Search';
import Price from 'components/Price/Price';

import cx from './CalendarPriceDay.scss';

interface ICalendarPriceDayProps {
    price?: ICalendarPrice | ICalendarEmptyPrice;
    isDisable: boolean;
}

const CalendarPriceDay: FC<ICalendarPriceDayProps> = props => {
    const {price, isDisable} = props;

    if (!price) {
        return null;
    }

    if (isDisable) {
        return <div className={cx('price')} />;
    }

    const hasPrice = isCalendarDayWithPrice(price);

    if (hasPrice) {
        return (
            <div
                className={cx('price', {
                    price_isLow: price.isLow,
                })}
            >
                <Price
                    isRound
                    isRoughly={price.roughly}
                    value={price.value}
                    currency={CURRENCY_RUB}
                />
            </div>
        );
    }

    const {emptyPriceReason} = price;

    switch (emptyPriceReason) {
        case ECalendarEmptyPriceReason.SOLD_OUT: {
            return (
                <div className={cx('price')}>
                    {i18nPriceCalendarBlock.soldOutDay()}
                </div>
            );
        }

        case ECalendarEmptyPriceReason.NO_DIRECT_TRAINS: {
            return <div className={cx('price')}>{CHAR_WIDE_MIDDLE_DASH}</div>;
        }

        case ECalendarEmptyPriceReason.OTHER: {
            return (
                <div className={cx('price')}>
                    <SearchIcon
                        className={cx('searchIcon')}
                        width={12}
                        height={12}
                    />
                </div>
            );
        }
    }

    return null;
};

export default CalendarPriceDay;
