import get from 'lodash/get';
import moment from 'moment';

import {ECalendarType, ICalendarDay, TCalendarMask} from '../types';

import {isHoliday} from 'utilities/dateUtils/isHoliday';
import getScrolledMoment from 'components/Calendar/utilities/getScrolledMoment';

import isAfterDay from './isAfterDay';
import getUTCDateByParams from './getUTCDateByParams';
import getUTCDateByDate from './getUTCDateByDate';
import isSameDay from './isSameDay';
import isBeforeDay from './isBeforeDay';
import isBetweenDay from './isBetweenDay';
import isSaturdayOrSunday from './isSaturdayOrSunday';

function isDisabledByMask(
    dayParams: ICalendarDay | false,
    mask?: TCalendarMask,
): boolean {
    if (!dayParams || !mask) {
        return false;
    }

    const {year, month, day} = dayParams;

    return !get(mask, [year, month + 1, day - 1].join('.'));
}

type TCalendarState = {
    day: ICalendarDay | false;
    mask?: TCalendarMask;
    nowDate: Date;
    minDate: Date;
    maxDate: Date;
    endDate: Date;
    startDate: Date;
    hoveredDate?: Date;
    calendarType?: ECalendarType;
    newExperimentalCalendar?: boolean;
    canSelectRange?: boolean;
};

export type TCalendarDaySettings = {
    isWeekend: boolean;
    isHovered: boolean;
    isStartDate: boolean;
    isEndDate: boolean;
    isScrolledDate: boolean;
    isRangeDay: boolean;
    isActive?: boolean;
    isDisable: boolean;
};

export default (calendarState: TCalendarState): TCalendarDaySettings => {
    const {
        day,
        mask,
        nowDate,
        minDate,
        maxDate,
        endDate,
        startDate,
        hoveredDate,
        calendarType,
        newExperimentalCalendar,
        canSelectRange,
    } = calendarState;

    let isHovered = false;
    let isStartDate = false;
    let isEndDate = false;
    let isRangeDay = false;
    let isWeekend = false;
    let isDisable = true;
    let isScrolledDate = false;

    const isMasked = isDisabledByMask(day, mask);

    if (day) {
        const currentDayDate = getUTCDateByParams(day, nowDate);
        const preparedMinDate = getUTCDateByDate(minDate);
        const preparedMaxDate = getUTCDateByDate(maxDate);
        const momentDay = moment(new Date(day.year, day.month, day.day));

        const dayIsSaturdayOrSunday = isSaturdayOrSunday(currentDayDate);
        const dayIsHoliday = isHoliday(momentDay);

        isWeekend = dayIsHoliday ?? dayIsSaturdayOrSunday;
        isDisable =
            momentDay.isBefore(preparedMinDate, 'day') ||
            momentDay.isAfter(preparedMaxDate, 'day');

        isScrolledDate =
            getScrolledMoment(startDate, endDate, preparedMinDate)?.isSame(
                momentDay,
                'day',
            ) ?? false;

        if (hoveredDate) {
            isHovered = isSameDay(currentDayDate, hoveredDate);
        }

        if (newExperimentalCalendar) {
            if (startDate) {
                // Дата является начальной, если
                // выбрано две даты и она совпадает с начальном
                if (endDate) {
                    isStartDate = isSameDay(currentDayDate, startDate);
                } else {
                    // Если выбрана одна дата, она совпадает с начальной и ховера нет, либо он правее даты
                    isStartDate =
                        isSameDay(currentDayDate, startDate) &&
                        (!hoveredDate ||
                            !canSelectRange ||
                            isAfterDay(hoveredDate, currentDayDate));
                }

                // Дата под ховером и левее выбранной даты (инверсия ренджей)
                if (hoveredDate && isHovered && !endDate && canSelectRange) {
                    isStartDate = isBeforeDay(hoveredDate, startDate);
                }

                // Дата является конечной, если
                // Отмечена как стратовая, но есть ховер левее
                isEndDate = Boolean(
                    hoveredDate &&
                        isSameDay(currentDayDate, startDate) &&
                        isBeforeDay(hoveredDate, currentDayDate),
                );

                // Дата является ховером правее стартовой даты
                if (hoveredDate && isHovered) {
                    isEndDate = isAfterDay(hoveredDate, startDate);
                }
            }

            // Дата является конечной, если она явно выбрана
            if (endDate) {
                isEndDate = isSameDay(currentDayDate, endDate);
            }

            // Дата попадает во внутрь ренджа,
            // если ховер правее стартовой даты и она между ховер и стартом
            // или ховер левее стартовой даты и она между
            // дата конца в обоих случаях не выбрана
            if (startDate && hoveredDate && !endDate) {
                isRangeDay =
                    (isAfterDay(hoveredDate, startDate) &&
                        isBetweenDay(currentDayDate, startDate, hoveredDate)) ||
                    (isBeforeDay(hoveredDate, startDate) &&
                        isBetweenDay(currentDayDate, hoveredDate, startDate));
            }

            // если явно выбрано начало и конец, дата находится между ними
            if (endDate) {
                isRangeDay = isBetweenDay(currentDayDate, startDate, endDate);
            }

            // Если промежуток выбирать нельзя, то
            isRangeDay = Boolean(canSelectRange) && isRangeDay;
            isEndDate = Boolean(canSelectRange) && isEndDate;
        } else {
            if (startDate) {
                isStartDate = isSameDay(currentDayDate, startDate);
            }

            if (endDate) {
                isEndDate = isSameDay(currentDayDate, endDate);
            }

            if (calendarType === ECalendarType.START_DATE) {
                if (endDate) {
                    if (startDate) {
                        isRangeDay = isBetweenDay(
                            currentDayDate,
                            startDate,
                            endDate,
                        );
                    } else if (hoveredDate) {
                        isRangeDay = isBetweenDay(
                            currentDayDate,
                            hoveredDate,
                            endDate,
                        );
                    }
                }
            } else if (startDate) {
                if (hoveredDate) {
                    isRangeDay = isBetweenDay(
                        currentDayDate,
                        startDate,
                        hoveredDate,
                    );
                }

                if (endDate) {
                    isRangeDay = isBetweenDay(
                        currentDayDate,
                        startDate,
                        endDate,
                    );
                } else if (isHovered) {
                    isRangeDay = true;
                }

                isDisable = isDisable || isBeforeDay(currentDayDate, startDate);
            }
        }
    }

    return {
        isWeekend,
        isHovered,
        isStartDate,
        isEndDate,
        isScrolledDate,
        isRangeDay: isRangeDay || isStartDate || isEndDate,
        isActive: mask && !isMasked,
        isDisable: isMasked || isDisable,
    };
};
