import React, {
    MouseEventHandler,
    PureComponent,
    ReactNode,
    RefObject,
} from 'react';

import {IWithClassName} from 'types/IWithClassName';
import {IWithDeviceType} from 'types/IWithDeviceType';
import {ECalendarType} from 'components/Calendar/types';

import {deviceModMobile} from 'utilities/stylesUtils';
import getPreparedDateForTrigger from 'components/Calendar/utilities/getPreparedDateForTrigger';
import {mergeRefs} from '@yandex-lego/components/lib/mergeRefs';

import {TDatePickerSize} from 'components/DatePicker/DatePicker';
import SearchControl from 'components/SearchControl/SearchControl';
import CalendarIcon from 'icons/16/Calendar';
import DatePickerIcon from 'icons/16/DatePicker';
import CloseCircleIcon from 'icons/16/CloseCircle';

import cx from './DatePickerTrigger.scss';

export interface IDatePickerTriggerProps
    extends IWithClassName,
        IWithDeviceType {
    focusClassName: string;
    calendarType: ECalendarType;
    placeholder: string;
    visiblePopup: boolean;
    date: Nullable<Date>;
    triggerViewType: 'union' | 'tile';
    triggerButtonName?: string;
    withThinBorder?: boolean;
    size: TDatePickerSize;
    isShortFormatDate: boolean;
    isMobile: boolean;
    canRenderResetIcon: boolean;
    tabIndex?: number;
    hasError: boolean;
    triggerValue?: string;
    onClick?: () => void;
    onMouseDown?: () => void;
    onFocus?: () => void;
    onBlur?: () => void;
    onKeyDown?: () => void;
    onClickReset?: (type: ECalendarType) => void;
    triggerRef?: RefObject<HTMLElement>;
}

class DatePickerTrigger extends PureComponent<IDatePickerTriggerProps> {
    readonly buttonRef = React.createRef<HTMLButtonElement>();
    readonly labelRef = React.createRef<HTMLLabelElement>();

    static readonly defaultProps: Partial<IDatePickerTriggerProps> = {
        className: '',
        focusClassName: '',
        placeholder: '',
        triggerViewType: 'union',
        size: 'xl',
        calendarType: ECalendarType.START_DATE,
        visiblePopup: false,
        isShortFormatDate: true,
        canRenderResetIcon: true,
        date: null,
        isMobile: false,
        hasError: false,
    };

    focus(): void {
        const {current} = this.buttonRef;

        if (current) {
            current.focus();
        }
    }

    blur(): void {
        const {current} = this.buttonRef;

        if (current) {
            current.blur();
        }
    }

    private handleClickReset: MouseEventHandler = e => {
        const {onClickReset, calendarType} = this.props;

        onClickReset?.(calendarType);

        e.preventDefault();
        e.stopPropagation();

        this.focus();
    };

    private renderValue(): ReactNode {
        const {placeholder, size} = this.props;

        const valueNode = this.renderValueNode();

        if (size === 'xl') {
            return (
                <>
                    {this.renderButton()}
                    <SearchControl
                        placeholderClassName={cx('placeholder')}
                        placeholder={placeholder}
                        controlNode={valueNode}
                        isEmpty={!valueNode}
                    />
                </>
            );
        }

        return this.renderButton(valueNode || this.renderPlaceholderNode());
    }

    private renderValueNode(): ReactNode {
        const {date, triggerValue, isShortFormatDate} = this.props;

        const isEmptyValue = !date && !triggerValue;

        return (
            !isEmptyValue && (
                <div className={cx('value')}>
                    {triggerValue ||
                        getPreparedDateForTrigger(date as Date, {
                            isShortFormatDate,
                        })}
                </div>
            )
        );
    }

    private renderPlaceholderNode(): ReactNode {
        const {placeholder} = this.props;

        return <span className={cx('placeholder')}>{placeholder}</span>;
    }

    private renderResetIcon(): ReactNode {
        const {date, canRenderResetIcon, triggerValue} = this.props;

        if (canRenderResetIcon && (date || triggerValue)) {
            return (
                <div
                    className={cx('resetIcon')}
                    onClick={this.handleClickReset}
                >
                    <CloseCircleIcon
                        className={cx('crossIcon')}
                        width={18}
                        height={18}
                    />
                </div>
            );
        }

        return null;
    }

    private renderDatePickerIcon(): ReactNode {
        const {canRenderResetIcon, visiblePopup, date, triggerValue, size} =
            this.props;

        if (canRenderResetIcon && (date || triggerValue)) {
            return null;
        }

        if (size === 'xl') {
            return (
                <div className={cx('datePickerIcon')}>
                    <DatePickerIcon
                        className={cx('datePickerIcon__icon', {
                            active: visiblePopup,
                        })}
                    />
                </div>
            );
        }

        return <CalendarIcon className={cx('datePickerIcon__icon')} />;
    }

    private renderButton(buttonContent?: ReactNode): ReactNode {
        const {
            onFocus,
            onBlur,
            onKeyDown,
            tabIndex,
            triggerRef,
            triggerButtonName,
        } = this.props;

        return (
            <button
                className={cx('datePickerTriggerButton')}
                name={triggerButtonName}
                type="button"
                tabIndex={tabIndex}
                onFocus={onFocus}
                onBlur={onBlur}
                onKeyDown={onKeyDown}
                ref={mergeRefs(this.buttonRef, triggerRef)}
            >
                {buttonContent}
            </button>
        );
    }

    render(): ReactNode {
        const {
            className,
            visiblePopup,
            deviceType,
            calendarType,
            triggerViewType,
            withThinBorder,
            focusClassName,
            hasError,
            size,
            onClick,
            onMouseDown,
        } = this.props;

        return (
            <label
                className={cx(
                    'datePickerTrigger',
                    `datePickerTrigger_${size}`,
                    {
                        datePickerTrigger_focus: visiblePopup,
                        datePickerTrigger_error: hasError,
                        datePickerTrigger_withThinBorder: withThinBorder,
                        [`datePickerTrigger_viewType_${triggerViewType}`]:
                            triggerViewType,
                        [`datePickerTrigger_type_${calendarType}`]:
                            calendarType,
                    },
                    deviceModMobile('datePickerTrigger', deviceType),
                    className,
                    visiblePopup && focusClassName,
                )}
                ref={this.labelRef}
                onClick={onClick}
                onMouseDown={onMouseDown}
            >
                {this.renderValue()}
                {this.renderDatePickerIcon()}
                {this.renderResetIcon()}
            </label>
        );
    }
}

export default DatePickerTrigger;
