import React, {useRef} from 'react';

import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {deviceMods} from 'utilities/stylesUtils';

import InfoIcon from 'icons/16/Info';
import BottomSheet from 'components/BottomSheet/BottomSheet';
import MessageBoxPopup, {
    IMessageBoxPopupProps,
} from 'components/MessageBoxPopup/MessageBoxPopup';

import cx from './InfoPopup.scss';

const POPUP_POSITIONS = [
    EPopupDirection.TOP,
    EPopupDirection.BOTTOM,
    EPopupDirection.TOP_RIGHT,
    EPopupDirection.BOTTOM_RIGHT,
];

interface IInfoPopupProps {
    iconClassName?: string;
    contentClassName?: string;
    /**
     * Контейнер для тултипа
     * (актуально только в десктоп версии)
     */
    scopeRef?: React.RefObject<HTMLElement>;
    messageBoxProps?: Partial<IMessageBoxPopupProps>;
    className?: string;
}

const InfoPopup: React.FC<IInfoPopupProps> = props => {
    const {
        className,
        iconClassName,
        contentClassName,
        children,
        scopeRef,
        messageBoxProps,
    } = props;

    const deviceType = useDeviceType();
    const rootRef = useRef(null);
    const {value: isVisible, setTrue: show, setFalse: hide} = useBoolean(false);

    const button = (
        <button
            className={cx('root', className)}
            ref={rootRef}
            type="button"
            onClick={show}
        >
            <InfoIcon className={iconClassName} />
        </button>
    );

    const content = (
        <div
            className={cx(
                'content',
                deviceMods('content', deviceType),
                contentClassName,
            )}
        >
            {children}
        </div>
    );

    if (deviceType.isMobile) {
        return (
            <>
                {button}
                <BottomSheet
                    isOpened={isVisible}
                    onClose={hide}
                    popHistoryOnUnmount={false}
                >
                    {content}
                </BottomSheet>
            </>
        );
    }

    return (
        <>
            {button}
            <MessageBoxPopup
                isVisible={isVisible}
                anchorRef={rootRef}
                scopeRef={scopeRef}
                boundaryRef={scopeRef}
                direction={POPUP_POSITIONS}
                onClose={hide}
                {...messageBoxProps}
            >
                {content}
            </MessageBoxPopup>
        </>
    );
};

export default InfoPopup;
