import React, {useMemo} from 'react';

import {TTextColor} from 'components/Text/types';
import ValueType from 'types/currency/ValueType';
import {IWithClassName} from 'types/IWithClassName';
import {ECurrencyType} from 'types/currency/ECurrencyType';

import {formatPrice} from 'utilities/currency';

import * as i18nBlock from 'i18n/common-price';

import cx from './Price.scss';

export interface IPriceProps extends IWithClassName {
    /** Цвет */
    color?: TTextColor;

    /** Цена */
    value: ValueType;

    /** Валюта */
    currency: ECurrencyType;

    /** Показывать знак валюты */
    isCurrencyShown?: boolean;

    /**
     * Префикс "от"
     */
    isFrom?: boolean;

    /** Префикс "до" */
    isUpTo?: boolean;

    /** Не показывать цифры после запятой */
    isRound?: boolean;

    /** Примерная цена */
    isRoughly?: boolean;

    /** Префикс к цене */
    prefix?: string;
    prefixDelimiter?: string;

    /**
     * Нужно ли конвертировать валюту к национальной
     * Конвертация возможна только в случае наличия соответствующих данных
     */
    withConversion?: boolean;

    /** Постфикс к цене */
    postfix?: string;
}

const Price: React.FC<IPriceProps> = props => {
    const {
        className,
        color,
        value,
        currency,
        isCurrencyShown = true,
        isFrom,
        isUpTo,
        isRound,
        isRoughly,
        prefix,
        prefixDelimiter,
        postfix,
    } = props;

    let price = formatPrice({
        value,
        currency,
        isCurrencyShown,
        isRound,
        postfix,
    });

    price = isRoughly ? i18nBlock.roughly({price}) : price;

    const prefixNode = useMemo(() => {
        if (!prefix) {
            return null;
        }

        if (prefixDelimiter) {
            return `${prefix}${prefixDelimiter}`;
        }

        return <span className={cx('prefix')}>{prefix}</span>;
    }, [prefix, prefixDelimiter]);

    if (isFrom) {
        price = i18nBlock._from({price});
    } else if (isUpTo) {
        price = i18nBlock.upTo({price});
    }

    return (
        <span className={cx('root', className, color && `color_${color}`)}>
            {prefixNode}
            {price}
        </span>
    );
};

export default Price;
