import React, {Children, ReactElement, ReactNode} from 'react';

import RadioButtonSlideOption, {
    IRadioButtonSlideOptionProps,
} from 'components/RadioButtonSlide/components/RadioButtonSlideOption';

import cx from './RadioButtonSlide.scss';

export interface IRadioButtonSlideProps {
    children:
        | ReactElement<IRadioButtonSlideOptionProps>
        | ReactElement<IRadioButtonSlideOptionProps>[];
    value?: string;
    onChange?(value: string): void;
    disabled?: boolean;
    isLoading?: boolean;
    size?: 'm' | 'l';
    name?: string;
    backgroundClassName?: string;
    className?: string;
}

interface ITabPlaceholderStyles {
    width: number;
    left: number;
}

interface IRadioButtonSlideState {
    value: string;
    currentIndex: number;
    tabPlaceholderStyles: ITabPlaceholderStyles;
    itemWidth: number;
}

class RadioButtonSlide extends React.PureComponent<
    IRadioButtonSlideProps,
    IRadioButtonSlideState
> {
    state = {
        value: '',
        currentIndex: 0,
        tabPlaceholderStyles: {
            width: 0,
            left: 0,
        },
        itemWidth: 0,
    };

    rootRef = React.createRef<HTMLSpanElement>();

    componentDidMount(): void {
        const {value} = this.props;

        if (value) {
            this.setNewValue(value);
        } else {
            this.setState({
                tabPlaceholderStyles: this.getTabPlaceholderStyles(
                    this.state.currentIndex,
                ),
                itemWidth: this.itemWidth,
            });
        }
    }

    componentDidUpdate(prevProps: Readonly<IRadioButtonSlideProps>): void {
        const {value: prevValue} = prevProps;
        const {value} = this.props;
        const {value: stateValue} = this.state;

        if (value && value !== prevValue && value !== stateValue) {
            this.setNewValue(value);
        }
    }

    get itemWidth(): number {
        const {children} = this.props;
        let elementWidth = 0;

        if (this.rootRef.current) {
            elementWidth =
                this.rootRef.current.getBoundingClientRect().width /
                Children.count(children);
        }

        return elementWidth;
    }

    static Option = RadioButtonSlideOption;

    private setNewValue(value: string): void {
        const {children} = this.props;
        let newIndex = 0;

        Children.forEach(
            children as ReactElement<IRadioButtonSlideOptionProps>,
            (item, index) => {
                if (item.props.value === value) {
                    newIndex = index;
                }
            },
        );
        this.setState({
            value,
            currentIndex: newIndex,
            tabPlaceholderStyles: this.getTabPlaceholderStyles(newIndex),
            itemWidth: this.itemWidth,
        });
    }

    private getTabPlaceholderStyles(
        currentIndex: number,
    ): ITabPlaceholderStyles {
        const {children} = this.props;
        let width = this.itemWidth;
        let left = width * currentIndex;

        if (currentIndex === 0) {
            left += 1;
            width -= 1;
        }

        if (currentIndex === Children.count(children) - 1) {
            width -= 1;
        }

        return {width, left};
    }

    private handleItemClick =
        (
            index: number,
            itemCallback: IRadioButtonSlideOptionProps['onClick'],
        ) =>
        (value: string): void => {
            const {onChange} = this.props;

            this.setState({
                value,
                currentIndex: index,
                tabPlaceholderStyles: this.getTabPlaceholderStyles(index),
            });
            onChange?.(value);
            itemCallback?.(value);
        };

    render(): ReactNode {
        const {
            className,
            children,
            size = 'l',
            disabled,
            backgroundClassName,
        } = this.props;
        const {value, tabPlaceholderStyles, itemWidth} = this.state;

        return (
            <span
                ref={this.rootRef}
                className={cx('root', className, {
                    size_m: size === 'm',
                    size_l: size === 'l',
                    disabled,
                })}
                role="radio-group"
            >
                {Children.map(
                    children as ReactElement<IRadioButtonSlideOptionProps>,
                    (item, index) =>
                        React.cloneElement(item, {
                            ...item.props,
                            key: item.props.value,
                            className: cx('item', item.props.className, {
                                checked: item.props.value === value,
                                disabled: item.props.disabled || disabled,
                            }),
                            checked: item.props.value === value,
                            disabled: item.props.disabled || disabled,
                            onClick: this.handleItemClick(
                                index,
                                item.props.onClick,
                            ),
                            width: item.props.width || itemWidth,
                        }),
                )}
                <div className={cx('tab')} style={tabPlaceholderStyles} />
                <div className={cx('background', backgroundClassName)} />
            </span>
        );
    }
}

export default RadioButtonSlide;
