import React, {
    MutableRefObject,
    ReactNode,
    useCallback,
    useEffect,
    useMemo,
} from 'react';
import {isEmpty} from 'lodash';

import {IWithClassName} from 'types/IWithClassName';
import {
    ESearchFormBorderColor,
    ESearchFormFieldName,
    ESearchFormSize,
    ESearchFormStyleType,
    ESearchFormTriggerViewType,
    ISearchFormSuggests,
    ISearchFormValues,
    TSearchFormErrors,
} from 'components/SearchForm/types';
import {
    ICalendarPricesByType,
    TCalendarPrices,
} from 'types/common/calendarPrice/ICalendarPrice';
import typedReactMemo from 'types/typedReactMemo';
import {ECalendarType} from 'components/Calendar/types';
import {ISearchFormRenderTravelersBlockProps} from 'components/SearchForm/types/ISearchFormRenderTravelersBlockProps';
import {EWhenSpecialValues} from 'types/common/When';
import {ISelectItemPreparedOptions} from 'components/SearchForm/types/ISelectItemOptions';
import {EPubSubEvent} from 'types/EPubSubEvent';
import {TOnShowSuggests} from 'components/SearchSuggest/types/TOnShowSuggests';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {deviceMods} from 'utilities/stylesUtils';
import {publish} from 'utilities/pubSub';

import useStoredFields from 'components/SearchForm/hooks/useStoredFields';
import useFieldsRefsAndErrors from 'components/SearchForm/hooks/useFieldsRefsAndErrors';
import Suggest, {
    TNameSuggest,
} from 'components/SearchForm/components/Suggest/Suggest';
import ReverseDirectionButton from 'components/SearchForm/components/ReverseDirectionButton/ReverseDirectionButton';
import DatePicker from 'components/SearchForm/components/DatePicker/DatePicker';
import {
    IBaseSuggestItem,
    ISuggestValue,
    TSelectedValue,
} from 'components/SearchSuggest/SearchSuggest';
import useSearchSuggests from 'components/SearchForm/hooks/useSearchSuggests';
import ErrorTooltip from 'components/SearchForm/components/ErrorTooltip/ErrorTooltip';
import SubmitButton from 'components/SearchForm/components/SubmitButton/SubmitButton';
import {
    IDatePickerFooterProps,
    IDatePickerFooterPropsByCalendarType,
} from 'components/DatePicker/components/DatePickerFooter/DatePickerFooter';

import cx from './SearchForm.scss';

export interface IOnDateClickParams {
    startDate: Date | null;
    endDate?: Date | null;
    calendarType: ECalendarType;
}

export interface IBaseSearchFormProps {
    size?: ESearchFormSize;
    triggerViewType?: ESearchFormTriggerViewType;
    formStyleType?: ESearchFormStyleType;
    borderColor?: ESearchFormBorderColor;
    isExpanded?: boolean;
    initiallyCalendarIsOpen?: boolean;
    autoFocusFieldName?: ESearchFormFieldName;
    canToggleDropdowns?: boolean;
    formRef?: React.RefObject<HTMLFormElement>;
    isFullDateFormat?: boolean;
    validateOnMount?: boolean;
    canSubmitOnChange?: boolean;
    hideDateClearButton?: boolean;
    submitButtonText: string;
    fieldsRef?: MutableRefObject<IFormFieldsRef | undefined>;
    submitButtonClassName?: string;
    datePickerClassName?: string;
    datePickerTriggerClassName?: string;
    canAutoHideCalendar?: boolean;
    onInteract?(val: boolean): void;
    onDateClick?(args?: IOnDateClickParams): void;
    onShowCalendar?(calendarType: ECalendarType): void;
    onResetCalendar?(calendarType?: ECalendarType): void;
    onShowFieldPopup?(): void;
    onHideFieldPopup?(): void;
    onValidationFailed?(errors: TSearchFormErrors): void;
    onSuggestInputFocus?(fieldName: ESearchFormFieldName): void;
    resetRangeAfterFirstInteraction?: boolean;
}

export interface IFormFieldsRef {
    focusFieldByName: (fieldName: ESearchFormFieldName) => void;
}

interface ISearchFormProps<SuggestItem extends IBaseSuggestItem>
    extends IBaseSearchFormProps,
        IWithClassName {
    fromField?: ISuggestValue<SuggestItem>;
    withoutReverseDirection?: boolean;
    fromSearchSuggests?: SuggestItem[] | null;
    fromPointPlaceholder?: string;
    setFromPoint?(point: ISuggestValue<SuggestItem>): void;

    toField?: ISuggestValue<SuggestItem>;
    toSearchSuggests: SuggestItem[] | null;
    toPointPlaceholder: string;
    setToPoint?(point: ISuggestValue<SuggestItem>): void;

    startDateField: string | null;
    startDatePlaceholder: string;
    setStartDate(date: string | null): void;

    endDateField?: string | null;
    endDatePlaceholder?: string;
    setEndDate?(date: string | null): void;
    hasEndDate?: boolean;

    searchFormName: string;
    fieldsNames: ESearchFormFieldName[];
    canSubmitOnChange?: boolean;
    isRoundTrip?: boolean;
    hideDateClearButton?: boolean;
    previousFormValues?: PartialRecord<ESearchFormFieldName, SuggestItem[]>;
    calendarPrices?: TCalendarPrices;
    calendarPricesByType?: ICalendarPricesByType;
    defaultAutoSelectIndex?: number;
    isSuggestLoading?: boolean;
    validateForm(
        fieldsValues: ISearchFormValues,
        suggests: ISearchFormSuggests<SuggestItem>,
    ): TSearchFormErrors;
    setPreviousSearchFormValue?(data: any): void;
    uniqueSuggestValueName: keyof SuggestItem;
    getSuggestTitleAndDescription(item: SuggestItem): {
        title: string;
        description: string;
    };
    renderSuggestItemTitle?: (item: SuggestItem) => ReactNode;
    renderSuggestItemDescription?: (item: SuggestItem) => ReactNode;
    datePickerHeaderBlock?: React.ReactNode;
    datePickerFooterBlockParams?: IDatePickerFooterProps;
    datePickerFooterBlockParamsByType?: IDatePickerFooterPropsByCalendarType;
    specialWhenButtons?: EWhenSpecialValues[];
    renderTravellersBlock?(
        args: ISearchFormRenderTravelersBlockProps,
    ): React.ReactNode;
    requestCalendar?(type: ECalendarType): void;
    onReverseClick?(): void;
    onSubmit(args: {
        handleTrySubmit(): void;
        formErrors: TSearchFormErrors;
        setHasSubmitFailed(val: boolean): void;
    }): void;
    onSelectSuggestItem?(
        item: TSelectedValue<SuggestItem>,
        options?: ISelectItemPreparedOptions,
    ): void;
    fieldsRef?: MutableRefObject<IFormFieldsRef | undefined>;
    canFocusNextField?: (type: string, item: SuggestItem | false) => boolean;
    onFastDateClick?: (value: string) => void;
    sortSuggestItems?: (a: SuggestItem, b: SuggestItem) => number;
    onShowSuggests?: TOnShowSuggests<TNameSuggest, SuggestItem>;
}

const SearchForm = <SuggestItem extends IBaseSuggestItem>(
    props: ISearchFormProps<SuggestItem>,
): React.ReactElement => {
    const emptySuggest = useMemo((): SuggestItem[] => [], []);
    const emptyPreviousFormValues = useMemo(
        (): PartialRecord<ESearchFormFieldName, SuggestItem[]> => ({}),
        [],
    );

    const {
        className,
        submitButtonClassName,
        datePickerClassName,
        datePickerTriggerClassName,

        fromField,
        fromSearchSuggests,
        fromPointPlaceholder,
        setFromPoint,

        toField,
        toSearchSuggests,
        toPointPlaceholder,
        setToPoint,
        withoutReverseDirection = false,

        startDateField,
        startDatePlaceholder,
        setStartDate,

        endDateField,
        endDatePlaceholder,
        setEndDate,
        hasEndDate = false,

        validateOnMount,
        defaultAutoSelectIndex = 0,
        size = ESearchFormSize.XL,
        triggerViewType = ESearchFormTriggerViewType.UNION,
        formStyleType = ESearchFormStyleType.ROUNDED,
        fieldsNames,
        borderColor = ESearchFormBorderColor.YELLOW,
        isExpanded = true,
        initiallyCalendarIsOpen,
        autoFocusFieldName,
        submitButtonText,
        canSubmitOnChange,
        formRef,
        isFullDateFormat,
        searchFormName,
        validateForm,
        previousFormValues = emptyPreviousFormValues,
        calendarPrices,
        calendarPricesByType,
        setPreviousSearchFormValue,
        uniqueSuggestValueName,
        renderTravellersBlock,
        getSuggestTitleAndDescription,
        renderSuggestItemTitle,
        renderSuggestItemDescription,
        datePickerHeaderBlock,
        specialWhenButtons,
        canAutoHideCalendar,
        requestCalendar,
        isRoundTrip = false,
        hideDateClearButton = false,
        canToggleDropdowns = false,
        onDateClick,
        onInteract,
        onSuggestInputFocus,
        onReverseClick,
        onSubmit,
        onSelectSuggestItem,
        onShowCalendar,
        onShowFieldPopup,
        onHideFieldPopup,
        onValidationFailed,
        fieldsRef,
        canFocusNextField,
        isSuggestLoading,
        datePickerFooterBlockParams,
        datePickerFooterBlockParamsByType,
        onFastDateClick,
        resetRangeAfterFirstInteraction,
        onResetCalendar,
        sortSuggestItems,
        onShowSuggests,
    } = props;

    const deviceType = useDeviceType();

    const fromSuggests = useSearchSuggests(fromSearchSuggests);
    const toSuggests = useSearchSuggests(toSearchSuggests);

    const formValues = useMemo(() => {
        return {
            [ESearchFormFieldName.FROM]: fromField,
            [ESearchFormFieldName.TO]: toField,
            [ESearchFormFieldName.START_DATE]: startDateField,
            [ESearchFormFieldName.END_DATE]: endDateField,
        };
    }, [endDateField, fromField, startDateField, toField]);

    const formErrors = useMemo(() => {
        return validateForm(formValues, {
            [ESearchFormFieldName.FROM]: fromSuggests,
            [ESearchFormFieldName.TO]: toSuggests,
        });
    }, [formValues, fromSuggests, toSuggests, validateForm]);

    const {value: hasSubmitFailed, setTrue: setHasSubmitFailed} = useBoolean(
        validateOnMount ? !isEmpty(formErrors) : false,
    );

    const {
        handleTrySubmit,
        focusNextFieldByName,
        setFieldRefByName,
        getFieldRefByName,
        setErrorFieldRefByName,
        errorTooltipText,
        errorFieldRef,
        focusFieldByName,
    } = useFieldsRefsAndErrors({
        fieldsNames,
        autoFocusFieldName,
        formValues,
        canShowErrors: hasSubmitFailed,
        formErrors,
    });

    useEffect(() => {
        // Фокус выставляем только на десктопе, т.к. из-за модальности
        // в тачах легко можно будет потерять контекст
        if (validateOnMount && deviceType.isDesktop) {
            handleTrySubmit();
        }
    }, [validateOnMount, handleTrySubmit, deviceType.isDesktop]);

    useEffect(() => {
        if (fieldsRef) {
            fieldsRef.current = {
                focusFieldByName,
            };
        }
    }, [fieldsRef, focusFieldByName]);

    useEffect(() => {
        if (hasSubmitFailed) {
            onValidationFailed?.(formErrors);
        }
    }, [formErrors, hasSubmitFailed, onValidationFailed]);

    useEffect(() => {
        publish(EPubSubEvent.SEARCH_FORM_MOUNT);
    }, []);

    const {storeField, restoreField, storeFieldIfNotFilled} = useStoredFields({
        setFromPoint,
        setToPoint,
    });

    const submit = useImmutableCallback(() => {
        onSubmit({
            handleTrySubmit,
            formErrors,
            setHasSubmitFailed,
        });
    });

    const handleSubmit = useImmutableCallback(
        (e: React.FormEvent<HTMLFormElement>) => {
            e.preventDefault();

            submit();
        },
    );

    const handleSuggestInputFocus = useCallback(
        (fileName: ESearchFormFieldName) => {
            onSuggestInputFocus?.(fileName);
        },
        [onSuggestInputFocus],
    );

    const handleDateClick = useImmutableCallback(
        (args?: IOnDateClickParams) => {
            onDateClick?.(args);

            if (canSubmitOnChange) {
                requestAnimationFrame(() => {
                    submit();
                });
            }
        },
    );

    const handleShowCalendar = useImmutableCallback(
        (calendarType: ECalendarType) => {
            onShowFieldPopup?.();
            onShowCalendar?.(calendarType);
        },
    );

    const commonSuggestsProps = {
        triggerFocusClassName: cx('suggestTrigger_focus'),
        focusNextFieldByName,
        isExpanded,
        setErrorFieldRefByName,
        setFieldRefByName,
        getFieldRefByName,
        setPreviousSearchFormValue,
        uniqueValueName: uniqueSuggestValueName,
        storeField,
        getSuggestTitleAndDescription,
        renderSuggestItemTitle,
        renderSuggestItemDescription,
        onInteract,
        onInputFocus: handleSuggestInputFocus,
        restoreField,
        onSelectItem: onSelectSuggestItem,
    };

    const datePickerErrors = [
        ...(formErrors[ESearchFormFieldName.START_DATE] ?? []),
        ...(formErrors[ESearchFormFieldName.END_DATE] ?? []),
    ];

    return (
        <form
            className={cx(
                'searchForm',
                `searchForm_styleType_${formStyleType}`,
                `searchForm_triggerViewType_${triggerViewType}`,
                `searchForm_${size}`,
                withoutReverseDirection && 'searchForm_withoutReverseDirection',
                deviceMods('searchForm', deviceType),
                {
                    searchForm_rangeDates: hasEndDate,
                    searchForm_singleSuggest: fromField === undefined,
                    searchForm_withoutTravellers: Boolean(
                        !renderTravellersBlock,
                    ),
                    [`searchForm_borderColor_${borderColor}`]:
                        triggerViewType === ESearchFormTriggerViewType.UNION,
                },
                className,
            )}
            ref={formRef}
            onSubmit={handleSubmit}
        >
            {fromField !== undefined &&
                setFromPoint &&
                fromPointPlaceholder &&
                toField &&
                setToPoint && (
                    <>
                        <Suggest
                            className={cx('suggest')}
                            triggerClassName={cx(
                                'suggestTrigger',
                                'suggestTrigger_type_first',
                                {
                                    suggestTrigger_joined_first:
                                        fromField !== undefined && setFromPoint,
                                    suggestTrigger_withoutReverseDirection:
                                        withoutReverseDirection,
                                },
                            )}
                            searchFormName={searchFormName}
                            name={ESearchFormFieldName.FROM}
                            nextFieldName={ESearchFormFieldName.TO}
                            value={fromField}
                            error={
                                (hasSubmitFailed &&
                                    formErrors[
                                        ESearchFormFieldName.FROM
                                    ]?.[0]) ||
                                ''
                            }
                            otherField={toField.selectedValue}
                            placeholder={fromPointPlaceholder}
                            previousSuggestItems={
                                previousFormValues[ESearchFormFieldName.FROM]
                            }
                            suggestItems={fromSuggests || emptySuggest}
                            setPoint={setFromPoint}
                            canFocusNextField={canFocusNextField}
                            sortItems={sortSuggestItems}
                            onShowSuggests={onShowSuggests}
                            validateOnMount={validateOnMount}
                            {...commonSuggestsProps}
                        />

                        {!withoutReverseDirection && (
                            <ReverseDirectionButton
                                wrapperClassName={cx('reverseDirections')}
                                buttonClassName={cx('reverseDirectionsButton')}
                                storeField={storeField}
                                setToPoint={setToPoint}
                                setFromPoint={setFromPoint}
                                from={fromField}
                                to={toField}
                                onInteract={onInteract}
                                onClick={onReverseClick}
                            />
                        )}
                    </>
                )}

            {toField && setToPoint && (
                <Suggest
                    className={cx('suggest')}
                    triggerClassName={cx('suggestTrigger', {
                        suggestTrigger_type_first: fromField === undefined,
                        suggestTrigger_joined_second:
                            fromField !== undefined && setFromPoint,
                    })}
                    searchFormName={searchFormName}
                    name={ESearchFormFieldName.TO}
                    nextFieldName={ESearchFormFieldName.START_DATE}
                    value={toField}
                    error={
                        (hasSubmitFailed &&
                            formErrors[ESearchFormFieldName.TO]?.[0]) ||
                        ''
                    }
                    otherField={
                        fromField === undefined
                            ? false
                            : fromField.selectedValue
                    }
                    placeholder={toPointPlaceholder}
                    previousSuggestItems={
                        previousFormValues[ESearchFormFieldName.TO]
                    }
                    suggestItems={toSuggests || emptySuggest}
                    setPoint={setToPoint}
                    canFocusNextField={canFocusNextField}
                    isSuggestLoading={isSuggestLoading}
                    defaultAutoSelectIndex={defaultAutoSelectIndex}
                    sortItems={sortSuggestItems}
                    onShowSuggests={onShowSuggests}
                    validateOnMount={validateOnMount}
                    {...commonSuggestsProps}
                />
            )}

            <DatePicker
                className={cx('dates', datePickerClassName)}
                triggerClassName={cx(
                    'datePickerTrigger',
                    {lastControl: !renderTravellersBlock},
                    datePickerTriggerClassName,
                )}
                triggerFocusClassName={cx('datePickerTrigger_focus')}
                triggerViewType={triggerViewType}
                size={size}
                startDate={startDateField}
                error={
                    hasSubmitFailed &&
                    datePickerErrors.length > 0 &&
                    datePickerErrors
                }
                startDateError={
                    hasSubmitFailed &&
                    formErrors[ESearchFormFieldName.START_DATE]
                }
                endDate={endDateField}
                endDateError={
                    hasSubmitFailed && formErrors[ESearchFormFieldName.END_DATE]
                }
                hasEndDate={hasEndDate}
                canAutoHideCalendar={canAutoHideCalendar}
                storeFieldIfNotFilled={storeFieldIfNotFilled}
                isExpanded={isExpanded}
                calendarPrices={calendarPrices}
                calendarPricesByType={calendarPricesByType}
                onFastDateClick={onFastDateClick}
                initiallyCalendarIsOpen={initiallyCalendarIsOpen}
                startDatePlaceholder={startDatePlaceholder}
                endDatePlaceholder={endDatePlaceholder}
                headerBlock={datePickerHeaderBlock}
                footerBlockParams={datePickerFooterBlockParams}
                footerBlockParamsByType={datePickerFooterBlockParamsByType}
                specialWhenButtons={specialWhenButtons}
                setFieldRefByName={setFieldRefByName}
                setStartDate={setStartDate}
                setEndDate={setEndDate}
                setErrorFieldRefByName={setErrorFieldRefByName}
                requestCalendar={requestCalendar}
                isRoundTrip={isRoundTrip}
                canToggleDropdowns={canToggleDropdowns}
                hideDateClearButton={hideDateClearButton}
                isFullDateFormat={isFullDateFormat}
                onInteract={onInteract}
                onDateClick={handleDateClick}
                onShowCalendar={handleShowCalendar}
                onHideCalendar={onHideFieldPopup}
                onResetValue={onResetCalendar}
                resetRangeAfterFirstInteraction={
                    resetRangeAfterFirstInteraction
                }
            />

            {renderTravellersBlock?.({baseCx: cx, setFieldRefByName, submit})}

            <SubmitButton
                className={cx('submitButton', submitButtonClassName)}
                isSimple={
                    deviceType.isMobile ||
                    triggerViewType === ESearchFormTriggerViewType.TILE
                }
                label={submitButtonText}
            />

            <ErrorTooltip text={errorTooltipText} anchorRef={errorFieldRef} />
        </form>
    );
};

export default typedReactMemo(SearchForm);
