import React, {useEffect, useMemo, useRef} from 'react';
import moment from 'moment';

import {ROBOT} from 'constants/dateFormats';

import {
    ICalendarPricesByType,
    TCalendarPrices,
} from 'types/common/calendarPrice/ICalendarPrice';
import {
    ESearchFormFieldName,
    ESearchFormSize,
    ESearchFormTriggerViewType,
} from 'components/SearchForm/types';
import {IWithClassName} from 'types/IWithClassName';
import {ECalendarType} from 'components/Calendar/types';
import {EWhenSpecialValues} from 'types/common/When';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getUTCDateByDate from 'utilities/dateUtils/getUTCDateByDate';
import addYears from 'components/Calendar/utilities/addYears';
import getPreparedDateForTrigger from 'components/Calendar/utilities/getPreparedDateForTrigger';
import {deviceModMobile} from 'utilities/stylesUtils';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getFormattedDate from 'components/SearchForm/components/DatePicker/utilities/getFormattedDate';
import isSpecialWhen from 'utilities/dateUtils/when/isSpecialWhen';
import getWhenSpecialValueText from 'utilities/dateUtils/when/getWhenSpecialValueText';

import CommonDatePicker, {
    DatePicker as CommonDatePickerClass,
} from 'components/DatePicker/DatePicker';
import DatepickerToolbar from 'components/SearchForm/components/DatePicker/components/DatepickerToolbar/DatepickerToolbar';
import {
    IDatePickerFooterProps,
    IDatePickerFooterPropsByCalendarType,
} from 'components/DatePicker/components/DatePickerFooter/DatePickerFooter';

import cx from './DatePicker.scss';

interface IDatePickerProps extends IWithClassName {
    triggerClassName: string;
    triggerFocusClassName: string;
    triggerViewType: ESearchFormTriggerViewType;
    size: ESearchFormSize;
    isExpanded?: boolean;
    headerBlock?: React.ReactNode;
    canAutoHideCalendar?: boolean;
    footerBlockParams?: IDatePickerFooterProps;
    footerBlockParamsByType?: IDatePickerFooterPropsByCalendarType;
    startDate: string | null;
    startDateError?: string[] | boolean;
    error: string[] | boolean;
    startDatePlaceholder: string;
    setStartDate(startDate: string | null): void;
    endDate?: string | null;
    endDateError?: string[] | boolean;
    endDatePlaceholder?: string;
    setEndDate?(startDate: string | null): void;
    hasEndDate: boolean;
    calendarPrices: TCalendarPrices | undefined;
    calendarPricesByType?: ICalendarPricesByType;
    setErrorFieldRefByName: any;
    initiallyCalendarIsOpen?: boolean;
    specialWhenButtons?: EWhenSpecialValues[];
    setFieldRefByName: any;
    isRoundTrip: boolean;
    storeFieldIfNotFilled(typeAndValue: {
        type: ESearchFormFieldName.START_DATE;
        value: string | null;
    }): void;
    requestCalendar?(type?: ECalendarType): void;
    onInteract?(val: boolean): void;
    onDateClick?(args?: {
        startDate: Date | null;
        endDate?: Date | null;
        calendarType?: ECalendarType;
    }): void;
    onResetValue?(calendarType?: ECalendarType): void;
    onFastDateClick?: (value: string) => void;
    hideDateClearButton: boolean;
    canToggleDropdowns: boolean;
    onShowCalendar?(calendarType: ECalendarType): void;
    onHideCalendar?(): void;
    isFullDateFormat?: boolean;
    resetRangeAfterFirstInteraction?: boolean;
}

const MIN_DATE = getUTCDateByDate(new Date());
const MAX_DATE = addYears(MIN_DATE, 1);
const NOW_DATE = new Date();

const DatePicker: React.FC<IDatePickerProps> = props => {
    const {
        className,
        triggerClassName,
        triggerFocusClassName,
        size,
        triggerViewType,
        headerBlock: headerBlockFromProps,
        footerBlockParams,
        footerBlockParamsByType,
        canAutoHideCalendar,
        startDate,
        startDateError,
        startDatePlaceholder,
        setStartDate,
        endDate,
        endDateError,
        endDatePlaceholder,
        setEndDate,
        hasEndDate,
        isRoundTrip,
        isExpanded,
        setErrorFieldRefByName,
        setFieldRefByName,
        storeFieldIfNotFilled,
        initiallyCalendarIsOpen,
        calendarPrices,
        calendarPricesByType,
        requestCalendar,
        specialWhenButtons,
        hideDateClearButton,
        canToggleDropdowns,
        isFullDateFormat,
        onDateClick,
        onInteract,
        onShowCalendar,
        onHideCalendar,
        error,
        onFastDateClick,
        resetRangeAfterFirstInteraction,
        onResetValue,
    } = props;

    const deviceType = useDeviceType();

    const datePickerInstanceRef = useRef<CommonDatePickerClass | null>(null);

    const setDatePickerInstance = useImmutableCallback(
        (node: CommonDatePickerClass) => {
            datePickerInstanceRef.current = node;

            if (!node) {
                return;
            }

            const calendarRefs = [setFieldRefByName, setErrorFieldRefByName];
            const {startDateTriggerRef, endDateTriggerRef} =
                node as CommonDatePickerClass;
            const calendarRefNames = {
                [ESearchFormFieldName.START_DATE]: startDateTriggerRef,
                [ESearchFormFieldName.END_DATE]: endDateTriggerRef,
            };

            Object.entries(calendarRefNames).forEach(
                ([refName, triggerRef]) => {
                    calendarRefs.forEach(calendarRef => {
                        const buttonNode =
                            triggerRef?.current?.buttonRef?.current;

                        if (buttonNode) {
                            calendarRef(refName)(buttonNode);
                        }
                    });
                },
            );
        },
    );

    const storeStartDate = useImmutableCallback(() => {
        storeFieldIfNotFilled({
            type: ESearchFormFieldName.START_DATE,
            value: startDate,
        });
    });

    const handleFastLinkClick = useImmutableCallback((value: string) => {
        storeStartDate();

        setStartDate(value);

        onDateClick?.();

        onFastDateClick?.(value);

        datePickerInstanceRef.current?.hidePopup();
    });

    const handleCalendarInteraction = useImmutableCallback(
        (type: ECalendarType) => {
            requestCalendar?.(type);
            onShowCalendar?.(type);
        },
    );

    const handleResetValue = useImmutableCallback(
        (calendarType: ECalendarType) => {
            handleCalendarInteraction(calendarType);
            onResetValue?.(calendarType);
        },
    );

    const triggerValue = useMemo(() => {
        if (hasEndDate) {
            return undefined;
        }

        if (startDate === null) {
            return '';
        }

        if (isSpecialWhen(startDate)) {
            return getWhenSpecialValueText(startDate);
        }

        return getPreparedDateForTrigger(new Date(startDate), {
            isShortFormatDate: false,
        });
    }, [hasEndDate, startDate]);

    const handleDaySelect = useImmutableCallback(
        ({
            startDate: selectedStartDate,
            endDate: selectedEndDate,
        }: {
            startDate: Date | null;
            endDate?: Date | null;
            calendarType: ECalendarType;
        }) => {
            storeStartDate();

            setStartDate(
                selectedStartDate
                    ? moment.utc(selectedStartDate).format(ROBOT)
                    : null,
            );

            setEndDate?.(
                selectedEndDate
                    ? moment.utc(selectedEndDate).format(ROBOT)
                    : null,
            );

            // Очень странный хак, потому что в текущей реализации формы
            // при выборе даты в календаре не происходит onBlur с кнопки дэйтпикера.
            onInteract?.(false);

            onDateClick?.({
                startDate: selectedStartDate,
                endDate: selectedEndDate,
            });
        },
    );

    const handleBlurDatePicker = useImmutableCallback(() => {
        onInteract?.(false);
    });

    const handleFocusDatePicker = useImmutableCallback(() => {
        onInteract?.(true);
    });

    const formattedStartDate = useMemo(() => {
        return getFormattedDate(startDate);
    }, [startDate]);

    const formattedEndDate = useMemo(() => {
        if (!endDate) {
            return null;
        }

        return getFormattedDate(endDate);
    }, [endDate]);

    useEffect(() => {
        if (initiallyCalendarIsOpen) {
            datePickerInstanceRef.current?.showPopup(ECalendarType.START_DATE);
        }
    }, [initiallyCalendarIsOpen]);

    const headerBlock = useMemo(() => {
        if (headerBlockFromProps) {
            return headerBlockFromProps;
        }

        if (!specialWhenButtons) {
            return null;
        }

        return (
            <DatepickerToolbar
                onClick={handleFastLinkClick}
                specialWhenButtons={specialWhenButtons}
            />
        );
    }, [handleFastLinkClick, headerBlockFromProps, specialWhenButtons]);

    return (
        <CommonDatePicker
            className={cx(
                'root',
                deviceModMobile('root', deviceType),
                className,
            )}
            size={size}
            triggerViewType={triggerViewType}
            ref={setDatePickerInstance}
            triggerClassName={triggerClassName}
            triggerFocusClassName={triggerFocusClassName}
            headerNode={headerBlock}
            footerBlockParams={footerBlockParams}
            footerBlockParamsByType={footerBlockParamsByType}
            canAutoHideCalendar={canAutoHideCalendar}
            startDate={formattedStartDate}
            startDatePlaceholder={startDatePlaceholder}
            errorStartDate={startDateError}
            endDate={formattedEndDate}
            endDatePlaceholder={endDatePlaceholder}
            errorEndDate={endDateError}
            error={error}
            hasEndDate={hasEndDate}
            isRoundTrip={isRoundTrip}
            maxDate={MAX_DATE}
            minDate={MIN_DATE}
            nowDate={NOW_DATE}
            tabIndex={isExpanded ? 0 : -1}
            triggerValue={triggerValue}
            prices={calendarPrices}
            pricesByType={calendarPricesByType}
            isModalView={deviceType.isMobile}
            deviceType={deviceType}
            canRenderResetIcon={deviceType.isDesktop && !hideDateClearButton}
            isShortTriggerFormatDate={deviceType.isDesktop && !isFullDateFormat}
            canToggle={canToggleDropdowns}
            onShowCalendar={handleCalendarInteraction}
            onHideCalendar={onHideCalendar}
            onResetValue={handleResetValue}
            onDaySelect={handleDaySelect}
            onFocus={handleFocusDatePicker}
            onBlur={handleBlurDatePicker}
            resetRangeAfterFirstInteraction={resetRangeAfterFirstInteraction}
        />
    );
};

export default React.memo(DatePicker);
